package org.lsst.ccs.config;


import javax.persistence.MappedSuperclass;
import java.io.Serializable;
import java.util.*;
import javax.persistence.Access;
import javax.persistence.AccessType;
import javax.persistence.Column;

/**
 * This class represents a set of parameter that have been modified for a given subsystem.
* <P>
 *     <B>Important notice</B> : for the moment an active configuration profile has a name and a tag.
 *     This combination is unique amongst active profile. BUT this is not the case once data of this type
 *     has been "pushed in history" :
 *     <UL>
 *         <LI> in history there are many versions of the same config profile
 *         <LI> it is not guaranteed that the same config name may be re-used for another subsystem
 *     </UL>
 *     So these naming strategies should be re-evaluated after some experimental uses.
 * @author bamade
 */
// Date: 11/04/12
@MappedSuperclass
public abstract class ConfigProfile implements Serializable {
    /**
     * first time the profile was valid
     */
    private long startTimestamp ;//generated
    /**
     * valid limit. defaults to eternity except when the object is pushed in history.
     */
    //todo: get rid of that one and replace by abstract method
    private long endTimestamp = PackCst.STILL_VALID ;

    // Subsystem name is a duplicate  (it is already in the current susbsyem)
    // todo: something about it (but keys are a problem)

    private String subsystemName ;

    /**
     * Subsystem tag
     */
    private String tag ;
    
    /**
     * Name of the category this configProfile is dedicated to
     */
    private String categoryName;
    
    /**
     * the configuration name for this category
     */
    private String configNameForCategory;

    /**
     * this Profile cannot be used for a dynamic configuration switch
     */
    private boolean changingStaticData ;

    //TODO add version

    /**
     * the level (defaults to END USER)
     * see PackCst.
     */
    //TODO: normalize with Command levels
    private int level = PackCst.END_USER_LEVEL ;

    /**
     * comments
     */
    private String remarks ;

    /**
     * config with same name and tag: id of previous version
     */
    private long previousConfigID ;
    
    ///////////////////////////// CONSTRUCTORS


    protected ConfigProfile() {
    }

    protected ConfigProfile(String subsystemName, String tag, int level, String categoryName, String configName) {
        this.subsystemName = subsystemName ;
        this.tag = tag ;
        this.level = level;
        this.categoryName = categoryName;
        this.configNameForCategory = configName;
    }
    //TODO :copy constructor

    /////////////////////////// ACCESSORS/MUTATORS
    public abstract long getId() ;
    protected abstract void setId(long id) ;

    public abstract SubsystemDescription getSubsystemDescription() ;

    public abstract Set<? extends ParameterConfiguration> getModifiedParameters() ;
    
    public  abstract ParameterConfiguration temporaryChangeConfigurationValue(String  parameterPath, long time, String value, boolean checked)  ;
    /*
    // build path obejct
    ParameterPath path = ParameterPath.valueOf(parameterPath) ;
    // fetch the ParameterConfiguration
    ParameterConfiguration parameter = this.fetch(path) ;
    //TODO if null create a new one and put it in engineering mode
    if(parameter == null) return null ;
    //calls the other method
    temporaryChangeConfigurationValue(parameter, time, value);
    return parameter ;
    }
    */


    public boolean isReadOnly() {
        return getId() != 0L ;
    }

    @Column(name="name")
    @Access(AccessType.PROPERTY)
    public String getName() {
        return getCategoryName()+":" + getConfigName();
    }
    
    public void setName(String name){
    // Blank method : name is saved in the lob        
    }

    public boolean isChangingStaticData() {
        return changingStaticData;
    }

    void setChangingStaticData(boolean changingStaticData) {
        this.changingStaticData = changingStaticData;
    }

/* todo :fix
    public String getSubsystemName() {
        return getSubsystemDescription().getSubsystemName() ;
    }
    */

    public String getSubsystemName() {
        return subsystemName;
    }

    public void setSubsystemName(String subsystemName) {
        //System.out.println("---------------- changing name" + subsystemName);
        if(this.subsystemName== null || "".equals(this.subsystemName)) {
            this.subsystemName = subsystemName;
        }
    }

    public String getTag() {
        return tag;
    }

    void setTag(String tag) {
        this.tag = tag;
    }
    
    public String getCategoryName(){
        return categoryName;
    }
    
    public String getConfigName(){
        return configNameForCategory;
    }

    public long getStartTimestamp() {
        return startTimestamp;
    }

    protected void setStartTimestamp(long startTimestamp) {
        this.startTimestamp = startTimestamp;
    }

    public long getEndTimestamp() {
        return endTimestamp;
    }

    void setEndTimestamp(long endTimestamp) {
        this.endTimestamp = endTimestamp;
    }

    public int getLevel() {
        return level;
    }

     void setLevel(int level) {
        this.level = level;
    }

    public String getRemarks() {
        return remarks;
    }

    /**
     * no influence on behaviour.
     * @param remarks
     */
    public  void setRemarks(String remarks) {
        this.remarks = remarks;
    }

    /**
     * may return the id of a previous ConfigProfile (0L if none)
     * @return the id of a previous config profile
     */
    public long getPreviousConfigID() {
        return previousConfigID;
    }

    void setPreviousConfigID(long previousConfigID) {
        this.previousConfigID = previousConfigID;
    }
    
    //////////////////////////// IDENT METHODS

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof ConfigProfile)) return false;

        ConfigProfile that = (ConfigProfile) o;
        if (getId() != that.getId()) return false;
        if (!getName().equals(that.getName())) return false;
        String tag = getTag() ;
        if (tag != null ? !tag.equals(that.getTag()) : that.getTag() != null) return false;

        return true;
    }

    @Override
    public int hashCode() {
        long id = getId() ;
        int result = (int) (id ^ (id >>> 32));
        result = 31 * result + getName().hashCode();
        String tag = getTag() ;
        result = 31 * result + (tag != null ? tag.hashCode() : 0);
        return result;
    }
    @Override
    public String toString() {
        return "{" +
                "id=" + getId() + "; name=" + getName() +
                "(" + getSubsystemDescription().getId() +")"  +
                ";configurations=" + this.getModifiedParameters() +
                '}';
    }

    /////////////////////////// OTHER METHODS

    /**
     * Looks if the {@code ConfigProfile} object contains a modification for
     * the parameter defined by the specified {@code PathObject}
     * @param path
     * @return a {@code parameterConfiguration} that matches the specified
     * {@code PathObject}
     */
    protected ParameterConfiguration fetch(PathObject path) {
        for(ParameterConfiguration config: this.getModifiedParameters()) {
            if(config.getPath().equals(path.getPath())) {
                return config ;
            }
        }
        return null;
    }

    /**
     * after a failed repair gets all the failed Modifications
     * @return a list of {@code ParameterConfiguration} that have failed to be
     * modified.
     */
    public List<ParameterConfiguration> getAllReconfigurationFailures()  {
        List<ParameterConfiguration> res = new ArrayList<>() ;
        for(ParameterConfiguration config: this.getModifiedParameters()) {
            if(config.getReConfigurationFailure() != null ){
               res.add(config)  ;
            }
        }
        return res ;
    }

    /**
     * lists the failures as String.
     * @return a String representation of the configuration failures.
     */
    public List<String> reportFailures() {
        List<String> res = new ArrayList<>() ;
        for(ParameterConfiguration config: getAllReconfigurationFailures()) {
           res.add(config.getPath() + " : " + config.getReConfigurationFailure())  ;
        }
        return res ;
    }
    
    
    public abstract void addParameterConfiguration(String parameterName, String value);
    
    /**
     * registers a list of parameter configurations
     * @param parameterConfigurations
     * @throws ImmutableStateException if the object is already registered in database
     * @throws IllegalArgumentException if levels are incompatibles
     * @throws IllegalArgumentException if the corresponding parameter descriptions are not alive in the current
     * subsystem description
     */
    // TODO: check if Description not deprecated!
    public abstract void addParameterConfigurations(ParameterConfiguration... parameterConfigurations) ;

    /**
     * removes a list of parameter configurations
     * @param parameterConfigurations
     * @throws ImmutableStateException if operated on an object registered to the database
     */
    public abstract void removeParameterConfigurations(ParameterConfiguration... parameterConfigurations) ;

    /**
     * gives the value of a given parameter at a given date on a ConfigProfile object.
     * beware : the main goal of this method is to fetch a transient parameter set during
     * an engineering mode session. Otherwise it just returns the value of the parameter
     * without any time stamp check (it is supposed that the <TT>ConfigProfile</TT> object
     * was the one active at this date.
     * <P>
     * @param parameterPath
     * @param date
     * @return null if parameter not found
     */
    public  String getValueAt( String parameterPath, long date) {
        //???? if (this == null) return null;
        if (parameterPath == null) return null;
        ParameterPath path = ParameterPath.valueOf(parameterPath);
        return getValueAt(path, date ) ;
    }

    public  String getValueAt( ParameterPath path, long date) {
        ParameterConfiguration parmConfig = this.fetch(path);
        if (parmConfig == null) { // look if subsystem parameter default value
            SubsystemDescription subsystemDescription = this.getSubsystemDescription();
            ParameterDescription description = subsystemDescription.fetch(path);
            if (description != null) {
                return description.getParameterBase().getDefaultValue();
            }
            return null ;
        } else {
            return parmConfig.getValueAt(date);
        }
    }
    
    /**
     * 
     * @return true if the {@code ConfigProfile} object contains unsaved changes
     */
    public abstract boolean isDirty();
}
