package org.lsst.ccs.config;

import org.lsst.gruth.jutils.Constraints;
import javax.persistence.MappedSuperclass;
import java.io.Serializable;

/**
 * The base class for all Parameter descriptions
 */
@MappedSuperclass
public abstract class ParameterDescription implements Serializable, PathObject{

    /**
     * possible details. Immutable in Ghost data
     */
    protected /*@Nullable*/ String description ;
    private  /*@NotNull*/ ParameterBase parameterBase ;
    /**
     * a user-friendly name: it's up to the config "designer" to make sure it's a unique name
     * otherwise the "pathName" of the parameterBase will be used. Immutable in ghost Data
     */
    //TODO: derived field! Formula is too cumbersome! see hack in code
    private /*@NotNull*/ String simpleName ;
    /**
     * Added constraints to the type, may be null. Immutable in ghost data
     */
    private /*@Nullable*/ String constraints ;
    
    private String category;

    //TODO:  startDate, endDate, user, tag are normally part of SubsystemDescription do we duplicate?
    //TODO: harmonize with level in Commands
    /**
     * used for configuration "rights"; a designer may have
     * the right to modify a parameter and an end-user not.
     * when automatically generated from a subsytemdescription
     * the level of all modifiable parameters is set to 10,
     * a parameter which is modifiable by an end-user is  less than 5.
     * Immutable in Ghost Data.
     *
     */
    private int level = PackCst.DESIGNER_LEVEL;

    /**
     * "static" parameter: it can only be set for a constructor but cannot be
     * changed dynamically at runtime.
     */
    private boolean notModifiableAtRuntime;

    ////////////////////////////// CONSTRUCTORS

    protected ParameterDescription() {

    }

    /**
     * builds a default Description out of a ParameterBase which is described in the original setup.
     * @param parameterBase
     */
    protected ParameterDescription(ParameterBase parameterBase) {
        this.setParameterBase( parameterBase) ;
    }

    /**
     * copy constructor.
     * Beware: the id is copied so do not use to make a new Parameterdescription that is to be registered later.
     * @param other
     */
    protected ParameterDescription(ParameterDescription other) {
        //this(other.getParameterBase()); no will not work!
        this.setId(other.getId());
        this.setParameterBase( other.parameterBase) ;
        this.description = other.description ;
        this.simpleName = other.simpleName;
        this.constraints = other.constraints ;
        this.level = other.level ;
        this.category = other.category;
        this.notModifiableAtRuntime = other.notModifiableAtRuntime ;
    }


    //////////////////////////// ACCESSORS/MUTATORS

    public abstract long getId() ;

    protected abstract void setId(long id) ;

    public boolean isReadOnly() {
        return this.getId() != 0L ;
    }


    public ParameterBase getParameterBase() {
        return parameterBase;
    }

    protected void setParameterBase(ParameterBase parameterBase) {
        this.parameterBase = parameterBase;
        String codeName = parameterBase.getCodeName() ;
        if(codeName == null) codeName = "" ;
        this.simpleName = String.format("%s/%s/%s",
                parameterBase.getComponentName(),
                codeName,
                parameterBase.getParameterName()) ;
    }

    public String getDescription() {
        return description;
    }

    /**
     * this data is not essential to the logic so it is modifiable anyway.
     * @param description
     */
    public  void setDescription(String description) {
        this.description = description ;
    }

    public String getSimpleName() {
        return simpleName;
    }

    public void setSimpleName(String simpleName) {
        if(isReadOnly()) {
            throw new ImmutableStateException("simple name read Only") ;
        }
        // this is a hack : because simpleName is a derived field
        if(this.simpleName == null || "".equals(this.simpleName)) {
            this.simpleName = simpleName;
        }
    }

    public String getConstraints() {
        return constraints;
    }

    public void setConstraints(String constraints) {
        if(isReadOnly()) {
            throw new ImmutableStateException("Constraints read Only") ;
        }
        this.constraints = constraints;
    }
    
    public String getCategory(){
        return category;
    }
    
    public void setCategory(String category){
        if(isReadOnly()) {
            throw new ImmutableStateException("Category read Only") ;
        }
        this.category = category;
    }

    public int getLevel() {
        return level;
    }

    public void setLevel(int level) {
        if(isReadOnly()) {
            throw new ImmutableStateException("Level read Only") ;
        }
        this.level = level;
    }

    public boolean isNotModifiableAtRuntime() {
        return notModifiableAtRuntime;
    }

    public void setNotModifiableAtRuntime(boolean notModifiableAtRuntime) {
        this.notModifiableAtRuntime = notModifiableAtRuntime;
    }
    //////////////// IDENT METHODS
    // TODO: inclusion of id?

    /**
     * equals is special: it works with any Path Object!
     * @param o
     * @return
     */
    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof PathObject)) return false;

        PathObject that = (PathObject) o;

        if (!parameterBase.getPath().equals(that.getPath())) return false;

        return true;
    }

    @Override
    public int hashCode() {
        return parameterBase.getPath().hashCode();
    }

    @Override
    public String toString() {
        return "{" + getId() +
                ": base=" + parameterBase +
                '}';
    }


    public String getComponentName() {
        return parameterBase.getComponentName();
    }

    public String getCodeName() {
        return parameterBase.getCodeName();
    }

    public String getParameterName() {
        return parameterBase.getParameterName();
    }

    public String getDefaultValue() {
        return parameterBase.getDefaultValue() ;
    }

    @Override
    public ParameterPath getPath() {
        return parameterBase.getPath();
    }

    public String getTypeName() {
        return parameterBase.getTypeName();
    }

    public Object checkValue(String value) {
        return Constraints.check(this.getParameterBase().getTypeName(),
                value, this.getConstraints()) ;
    }
    /**
     * creates a String to be included in a .preperties file
     * @param value default value to be included in the text
     * @return
     */
    public String toPropertyString(String value, boolean commentOutValue) {
        if(value == null) {value ="" ;}
        StringBuilder builder = new StringBuilder() ;
        String pathName = parameterBase.getPath().toString() ;
        builder.append("\n#********  ").append(pathName) ;
        if(description != null) {
            String withPounds = description.replaceAll("\n","\n#") ;
            builder.append('\n').append("#** ").append(withPounds) ;
        }
        if(isNotModifiableAtRuntime()) {
            builder.append("\n#**  static parameter (not modifiable at runtime)") ;
        }
        if(constraints!= null && !"".equals(constraints.trim())) {
            builder.append("\n#** constraints : ").append(constraints).append(" ; type : ")
                    .append(TypeInfos.get( parameterBase.getTypeName())) ;
        } 
        else {
            builder.append("\n#** type : " )
                    .append(TypeInfos.get( parameterBase.getTypeName())) ;
        }
        if(category!= null && !"".equals(category.trim())) {
            builder.append("\n#** category : ").append(category);
        }
        String propName = this.simpleName;
        if( propName== null || "".equals(propName.trim())) {
            propName = pathName ;
        } else {
            propName = propName.trim().replaceAll(" ", "\\\\ ");
        }
        builder.append("\n#**\n#********\n\n").append(commentOutValue?"#":"").append(propName).append(" = ").append(value).append('\n') ;

        //System.out.println(" -> -> " +builder );
        return builder.toString() ;
    }

    /**
     * default value is included in the text that describes the parameter as a property
     * @return
     */
    public String toPropertyString() {
        return toPropertyString(getDefaultValue(), true) ;
    }
}
