package org.lsst.ccs.config;

import org.lsst.gruth.jutils.Constraints;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.framework.ConfigurationProxy;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.structs.ViewValue;
import org.lsst.gruth.types.TypeUtils;

import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.stream.Collectors;
import static java.util.stream.Collectors.toSet;
import javax.swing.tree.TreeNode;
import org.lsst.ccs.CCSCst;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.config.utilities.ConfigUtils;
import org.lsst.ccs.framework.ComponentLookupService;
import org.lsst.ccs.framework.Configurable;
import org.lsst.ccs.framework.ConfigurationServiceException;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.utilities.functions.Cocoon;
import org.lsst.gruth.jutils.DescriptiveNode;
import org.lsst.gruth.jutils.HollowParm;

/**
 * register locally changes in parameters (that is creates a new Properties "resource" for the
 * configuration)
 * <P/>
 * @author bamade
 */
// Date: 04/10/12

public class LocalConfigurationProxy implements ConfigurationProxy {
    
    private final static String FILE_SEPARATOR = System.getProperty("file.separator");
    public static final String DEFAULT_CONFIG_NAME = "";
    public static final String DEFAULT_CAT = "";

    
    /**
     * When locally registering properties the <TT>LocalConfigurationProxy</TT> may need to write
     * properties in a File or in memory for tests purposes.
     * Classes that support this interface are able to create
     * the corresponding resource.
     */
    public static interface WriterProvider {
        public PrintWriter getPrintWriter(String fileName) throws IOException;
        public Properties getConfigurationProperties(String configFileName) throws IOException;
        
    }
    private final ASubsystemDescription subsystemDescription ;
    
    private final String subsystemName;
    private final String tagName;
    /**
     * the configuration that is running in engineering/normal mode
     * (in normal mode both configurations are the same)
     */
    private final Map<String, AConfigProfile> configProfileMap = new HashMap<>();
    
    Logger logger = Logger.getLogger("org.lsst.ccs.config");
    
    private ComponentLookupService lookup;
    
    private ConfigurationState configState = ConfigurationState.UNCONFIGURED;
    
    private WriterProvider writerProvider = new WriterProvider() {
        @Override
        public PrintWriter getPrintWriter( String baseName ) throws IOException {
            
            //Check if the properties file already exists in the Bootstrap Environment
            String pathInBootstrap = BootstrapResourceUtils.getPathOfPropertiesFileInUserResourceDirectories(baseName);
            //If it does not exist...
            if ( pathInBootstrap == null ) {
                // then check if a there is a Bootstrap user defined directory in which to write it.
                String topMostUserDirectory = BootstrapResourceUtils.getTopUserResourceDirectory();
                //If there is not Bootstrap user Directory defined.....
                if ( topMostUserDirectory == null ) {
                    //Set the path to org.lsst.ccs.workdir
                    String workdir = BootstrapResourceUtils.getBootstrapSystemProperties().getProperty("org.lsst.ccs.workdir", "");
                    if ( !workdir.isEmpty() && ! workdir.endsWith(FILE_SEPARATOR)) {
                        workdir += FILE_SEPARATOR;
                    }
                    baseName = workdir+baseName;
                } else {
                    baseName = topMostUserDirectory+baseName;
                }
            } else {
                baseName = pathInBootstrap;
            }
            
            if ( ! baseName.endsWith(".properties") ) {
                baseName += ".properties";
            }
            
            logger.info("Saving configuration  to "+baseName);
            
            
            return new PrintWriter(baseName, "ISO-8859-1") ;
        }
        
        public Properties getConfigurationProperties(String configFileName) throws IOException {
            if ( ! configFileName.endsWith(".properties") ) {
                configFileName += ".properties";
            }
            Properties configProps = new Properties();
            InputStream propsIs = null;
            propsIs = BootstrapResourceUtils.getBootstrapResource(configFileName);
            if (propsIs == null){
                throw new IllegalArgumentException("Could not find configuration file : " + configFileName);
            }
            configProps.load(propsIs);
            return configProps;
        }
    } ;
    
    /**
     * Builds a configuration proxy out of a subsystem description.
     * @param subsystemDesc 
     */
    public LocalConfigurationProxy(SubsystemDescription subsystemDesc){
        this.subsystemDescription = (ASubsystemDescription)subsystemDesc;
        subsystemName = subsystemDescription.getSubsystemName();
        tagName = subsystemDescription.getTag();
        if(System.getProperty("org.lsst.ccs.testcontext","false").equals("true")){
            this.writerProvider = InMemoryWriterProvider.getInstance();
        }
    }
    
    // Ad hoc constructor
    public LocalConfigurationProxy(ConfigProfile configProfile){
        this(configProfile.getSubsystemDescription());
        configProfileMap.put(configProfile.getCategoryName(), (AConfigProfile)configProfile);
    }
    
    public Set<String> getCategorySet(){
        return subsystemDescription.getCategorySet();
    }
    
    /**
     * Builds a TreeNode representing the hierarchical structure of the worker subsystem.
     * This method is used to build a TreeNode modified with a configuration we want
     * a subsystem to be started with.
     * @param taggedCategories
     * @return 
     */
    public TreeNode getInitialDescriptiveNode(Map<String,String> taggedCategories) {
        DescriptiveNode modifiedDescriptiveNode = subsystemDescription.getTopComponentNode();
        /* From Here
        creates a Subsystem instance out of a Descriptive Node and Properties
        (that may override parameters of the descriptive node)
        */
        //these Configuration objects: SubsystemDescription and ConfigProfile should be handled by a DAO
        // right now we operate locally so it is a special dummy DAO: MemoryDAO
        CCSCst.LOG_TODO.fine("TODO: DAO at creation of SubsystemDescription should also be able to interact with remote database (switch between local and remote?)") ;
        /*
        since No SubsystemDescription can be used unless it is under the control
        of a DAO this description is passed to a dummy "MemoryDAO"
        */
        MemoryDAO dao = new MemoryDAO();
        dao.saveAbstractDescription(subsystemDescription);
        
        /*
        we create a <TT>ConfigProfile</TT> (where parameters  have values)
        out of the subsystem description
        This ConfigProfile is modified with the configuration Properties
        and saved to the MemoryDAO
        The modified configuration is saved to a ModifiedDescriptiveNode
        */
        // if no configuration create an empty configuration
        // TO remove
        boolean exceptionThrown = false;
        
        for(Map.Entry<String,String> entry : taggedCategories.entrySet()){
            Properties configProps = null;
            AConfigProfile profileForCat;
            try {
                configProps = loadPropertiesForCat(entry.getKey(),entry.getValue());
                profileForCat = createConfigProfileOutOfProperties(configProps, entry.getKey(), entry.getValue());
            } catch(IOException ex){
                exceptionThrown = true;
                profileForCat = createConfigProfileOutOfProperties(configProps, entry.getKey(), DEFAULT_CONFIG_NAME);
            }
            configProfileMap.put(profileForCat.getCategoryName(), profileForCat);
//            dao.saveAbstractConfig(newConfigProfile);
            configState = exceptionThrown ? ConfigurationState.UNCONFIGURED : ConfigurationState.CONFIGURED;
        }
        
        modifiedDescriptiveNode = getModifiedConfigurationData();
        
        return modifiedDescriptiveNode;
    }
    
    @Override
    public void setLookup(ComponentLookupService lookup) {
        this.lookup = lookup;
    }
    
    @Override
    public String getTagName() {
        return tagName;
    }
    
    /**
     * Builds a ConfigurationInfo object reflecting the configuration state and
     * ready to be sent on the buses.
     * @return
     */
    
    @Override
    public synchronized ConfigurationInfo buildConfigurationInfo(){
        Map<String,String> tags = new HashMap<>();
        Map<String, Boolean> hasCatChanges = new HashMap<>();
        ArrayList<ConfigurationParameterInfo> parametersView = new ArrayList<>();
        
        for (ConfigProfile profile : configProfileMap.values()){
            String category = profile.getCategoryName();
            // set tag for category
            tags.put(category, profile.getConfigName());
            hasCatChanges.put(profile.getCategoryName(), profile.isDirty());
            for (ParameterDescription parmDesc : subsystemDescription.getParamDescriptions().stream()
                    .filter(desc -> desc.getCategory().equals(category))
                    .collect(toSet())){
                ConfigurationParameterInfo configParameterInfo;
                ParameterConfiguration parmConfig = profile.fetch(parmDesc);
                if(parmConfig != null){
                    configParameterInfo = new ConfigurationParameterInfo(parmDesc.getPath().toString(), 
                            parmDesc.getCategory(), parmDesc.getDefaultValue(), 
                            parmConfig.getConfiguredValue(), parmConfig.getValueAt(PackCst.STILL_VALID), parmConfig.hasChanged());
                } else {
                    configParameterInfo = new ConfigurationParameterInfo(parmDesc.getPath().toString(), 
                            parmDesc.getCategory(), parmDesc.getDefaultValue(), 
                            parmDesc.getDefaultValue(), parmDesc.getDefaultValue(), false);
                }
                parametersView.add(configParameterInfo);
            }
        }
        return new ConfigurationInfo(configState, tagName, tags, hasCatChanges, parametersView);
    }
    
    @Override
    public ViewValue checkForParameterChange(String componentName, String parameterName, Object value) {
        String strValue ;
        if(value instanceof String){
            strValue = (String)  value ;
        } else {
            strValue = TypeUtils.stringify(value) ;
        }
        ParameterPath path = new ParameterPath(componentName,"",parameterName);
        ParameterDescription parameterDescription = subsystemDescription.fetch(path) ;
        if(null == parameterDescription) {
            throw new IllegalArgumentException("incoherent parameter name for " + parameterName + "-> "
                    //+ subsystemDescription.getParamDescriptions());
                    + subsystemName);
        }
        if(parameterDescription.isNotModifiableAtRuntime()) {
            throw new IllegalStateException(" parameter " + parameterName + " not modifiable at runtime");
        }
        Object res = parameterDescription.checkValue(strValue) ;
        return new ViewValue(strValue, res) ;
    }
    
    @Override
    public synchronized void notifyParameterChange(String componentName, String parameterName, String value) {
        ParameterPath parameterPath = new ParameterPath(componentName,"",parameterName);
        //if static
        String category = subsystemDescription.fetch(parameterPath).getCategory();
        configProfileMap.get(category).temporaryChangeConfigurationValue(parameterPath.toString(), System.currentTimeMillis(), value, true);
    }
    
    @Override
    public synchronized void notifyUncheckedParameterChange(String componentName, String parameterName, Object value) {
        ParameterPath parameterPath = new ParameterPath(componentName,"",parameterName);
        
        String strValue ;
        if( value instanceof String) {
            strValue = (String)  value ;
        } else {
            strValue = TypeUtils.stringify(value) ;
        }
        String category = subsystemDescription.fetch(parameterPath).getCategory();
        configProfileMap.get(category).temporaryChangeConfigurationValue(parameterPath.toString(), System.currentTimeMillis(), strValue, false);
    }
    
    /**
     * Tagged categories are saved under a new tag name.
     * Other parameters are left unchanged, the configuration context remains
     * active.
     * @param taggedCategories
     * @throws IOException
     */
    @Override
    public synchronized void saveChangesForCategoriesAs(Map<String,String> taggedCategories) throws ConfigurationServiceException {
        for(Map.Entry<String,String> taggedCategory : taggedCategories.entrySet()){
            try {
                PrintWriter printWriter = writerProvider.getPrintWriter(ConfigUtils.baseNameFromNames(taggedCategory.getValue(), tagName, taggedCategory.getKey()));
                for (ParameterConfiguration parameter : configProfileMap.get(taggedCategory.getKey()).getModifiedParameters()){
                    String currentValue = parameter.getValueAt(PackCst.STILL_VALID);
                    boolean commentOut = currentValue.equals(parameter.getDescription().getDefaultValue());
                    printWriter.println(parameter.getDescription().toPropertyString(currentValue, commentOut));
                }
                printWriter.flush();
                printWriter.close();
                saveModifications(taggedCategory.getKey(), taggedCategory.getValue());
            } catch(IOException ex){
                configState = ConfigurationState.UNCONFIGURED;
                throw new ConfigurationServiceException("configuration service unavailable", ex);
            }
        }
        configState = isDirty() ? ConfigurationState.DIRTY : ConfigurationState.CONFIGURED;
    }
    
    
    @Override
    public synchronized void saveChangesForCategories(Set<String> categories) throws ConfigurationServiceException {
        saveChangesForCategoriesAs(getTaggedCategoriesForCats(categories));
    }
    
    private Map<String,String> getTaggedCategoriesForCats(Set<String> categories){
        Map<String,String> res = new HashMap<>();
        for (String category : categories){
            res.put(category, configProfileMap.get(category).getConfigName());
        }
        return res;
    }
    
    /**
     * Modification occurring in one of the tagged categories are saved.
     * Other changes are left untouched.
     * The configuration context is stopped if there are no other ongoing changes
     * in other categories.
     * @param taggedConfiguration a list of categoryName:configurationName
     */
    private synchronized void saveModifications(String category, String configName){
        AConfigProfile oldProfile = configProfileMap.get(category);
        AConfigProfile newProfile = new AConfigProfile(oldProfile, "", PackCst.DESIGNER_LEVEL, false, configName);
        configProfileMap.put(category, newProfile);
    }
    
    @Override
    public Object getDefaultParameterValue(String componentName, String parameterName) {
        ParameterPath path = new ParameterPath(componentName,"",parameterName);
        ParameterDescription parameterDescription = subsystemDescription.fetch(path) ;
        
        String strValue = parameterDescription.getDefaultValue();
        String type = parameterDescription.getTypeName() ;
        Object res = Constraints.check(type,strValue, null) ;
        return res;
    }
    
    /**
     * Loads properties files for one category only
     * @param categoryName the category
     * @param configName the configuration name for the specified category
     * @throws IOException if the configuration does not exist, except if configName
     * is the default configuration.
     * @return
     */
    private synchronized Properties loadPropertiesForCat(String categoryName, String configName) throws IOException {
        Properties props = new Properties();
        try {
            props.putAll(writerProvider.getConfigurationProperties(ConfigUtils.baseNameFromNames(configName, tagName,categoryName)));
        } catch (IllegalArgumentException ex){
            if (!configName.equals(DEFAULT_CONFIG_NAME)){
                throw ex;
            }
        }
        return props;
    }
    
    public void setWriterProvider(WriterProvider writerProvider) {
        this.writerProvider = writerProvider;
    }
    
    public WriterProvider getWriterProvider(){
        return writerProvider;
    }
    
    private boolean isDirty(){
        return configProfileMap.values().stream()
                .map(ConfigProfile::isDirty)
                .reduce(false, (a,b) -> a||b);
    }
    
    @Override
    public void loadCategories(Map<String,String> taggedCategories) throws ConfigurationServiceException, Exception {
        if (taggedCategories.isEmpty()) return;
        for(Map.Entry<String,String> entry : taggedCategories.entrySet()){
            try {
                Properties configProps = loadPropertiesForCat(entry.getKey(), entry.getValue());
                AConfigProfile newConfigProfile = createConfigProfileOutOfProperties(configProps, entry.getKey(), entry.getValue());
                switchConfiguration(configProfileMap.get(entry.getKey()), newConfigProfile);
                configProfileMap.put(newConfigProfile.getCategoryName(), newConfigProfile);
                saveModifications(entry.getKey(), entry.getValue());
            } catch (IOException ex){
                configState = ConfigurationState.UNCONFIGURED;
                throw new ConfigurationServiceException("configuration service problem", ex);
            } catch( Exception ex){
                configState = ConfigurationState.DIRTY;
                throw ex;
            }
        }
        configState = isDirty() ? ConfigurationState.DIRTY : ConfigurationState.CONFIGURED;
    }
    
    @Override
    public void dropUnsavedChangesForCategories(Collection<String> categories) throws Exception {
        if (categories.isEmpty()) return ;
        //TODO rollback !!
        Module mainModule = (Module) lookup.getComponentByName("main");
        if (mainModule == null) {
            throw new IllegalArgumentException("no component named main!");
        }
        Map<String, List<ParameterConfiguration>> oldMap = getConfigurationMapForCat(categories);
        mainModule.proceduralWalk(null, Cocoon.consumer((Configurable configurable) ->
        { //begin lambda
            String name = configurable.getName();
            List<ParameterConfiguration> oldList = oldMap.get(name);
            // default confguration has an empty list
            // if parameters are suppressed from OldList and not in list
            if(oldList == null) {
                oldList = Collections.emptyList() ;
            }
            try {
                for (ParameterConfiguration parmConfig : oldList) {
                    if (!categories.contains(parmConfig.getDescription().getCategory())) continue;
                    ParameterPath path = parmConfig.getPath();
                    String parameterName = path.getParameterName();
                    String value = parmConfig.getConfiguredValue() ;
                    //TODO: code copy : modify
                    if (parmConfig.getDescription().isNotModifiableAtRuntime()) {
                        // Should never happen in this context
                        throw new IllegalArgumentException(path + "not modifiable at runtime");
                    }
                    change(configurable, parameterName, value);
                }
                
            } catch (Exception exc) {
                configState = ConfigurationState.DIRTY;
                throw exc;
            }
        } /*end lambda */) // end Cocoon
        );
        // reconstruct an array of tagged categories
        for(String cat : categories){
            saveModifications(cat, configProfileMap.get(cat).getConfigName());
        }
        configState = isDirty() ? ConfigurationState.DIRTY : ConfigurationState.CONFIGURED;
    }
    
    /**
     * Configuration switch from one profile to another.
     * The tree of modules is traveled :
     * - a parameter that is assigned a new value in newProfile is changed to this value.
     * - a parameter that is not assigned a new value in newProfile is changed to its default
     *   value if it was modified in oldProfile.
     * TODO : does not make sense if the profiles refer to different subsystemDescriptions or categories
     *
     * @param oldProfile profile still active in database
     * @param newProfile profile to be registered afterwards
     * @throws Exception could be that a parameter change is refused by code or that rollback failed afterwards
     */
    public void switchConfiguration(ConfigProfile oldProfile, ConfigProfile newProfile) throws Exception {
        // this is wrong  in engineering mode switch if both are static and have the same static data see code below
        /* wrong test
        if(newProfile.isChangingStaticData()) {
        throw new IllegalArgumentException(" Profile " + newProfile.getName() +
        " deals with static parameters and so cannot be loaded at runtime")  ;
        } */
        Module mainModule = (Module) lookup.getComponentByName("main");
        if (mainModule == null) {
            throw new IllegalArgumentException("no component named main!");
        }
        // name of module, list of parameters modified by newProfile for this module
        Map<String, List<ParameterConfiguration>> newChanges = newProfile.getConfigurationMap();
        // name of module, list of parameters modified by oldProfile for this module
        Map<String, List<ParameterConfiguration>> oldMap = oldProfile.getConfigurationMap();
        
        // Store the changed values in case of rollback
        List<String[]> changedValues = new ArrayList<>();
        // Travel of the tree of modules
        // the lambda throws an Exception so is "cocooned" (see utilities.Cocoon
        mainModule.proceduralWalk(null, Cocoon.consumer((Configurable configurable) ->
        {
            String name = configurable.getName();
            List<ParameterConfiguration> newList = newChanges.get(name);
            List<ParameterConfiguration> oldList = oldMap.get(name);
            if(oldList == null) {
                oldList = Collections.emptyList() ;
            }
            try {
                if (newList != null) {
                    // Changes described in newProfile
                    for (ParameterConfiguration parmConfig : newList) {
                        // if parmConfig was modified by oldProfile, it does not
                        // need to be set to its default value since newProfile
                        // defines a new value for it.
                        oldList.remove(parmConfig);
                        ParameterPath path = parmConfig.getPath();
                        String parameterName = path.getParameterName();
                        // The new value to assign
                        String value = parmConfig.getConfiguredValue();
                        String oldValue = oldProfile.getValueAt(path, PackCst.STILL_VALID);
                        //TODO: some problems if strings are not exactly the same! (though the values are equals! examples : maps , unordered lists, floating point)
                        // change the test
                        if (oldValue.equals(value)) {
                            continue;
                        }
                        if (parmConfig.getDescription().isNotModifiableAtRuntime()) {
                            // TODO : throw exception or ignore ?
                            throw new IllegalArgumentException(path + "not modifiable at runtime");
                        }
                        String[] histElement = new String[]{name, parameterName, oldValue};
                        
                        // change of parameter

                        change(configurable, parameterName, value);
                        //last pushed in front
                        changedValues.add(0, histElement);
                    }
                }
                // The parameters that remain in oldList are to be set back to
                // their default value
                for (ParameterConfiguration parmConfig : oldList) {
                    ParameterPath path = parmConfig.getPath();
                    String parameterName = path.getParameterName();
                    String defaultValue = parmConfig.getDescription().getDefaultValue() ;
                    String oldValue = oldProfile.getValueAt(path, PackCst.STILL_VALID);
                    if (parmConfig.getDescription().isNotModifiableAtRuntime()) {
                        // TODO : throw exception or ignore ?
                        throw new IllegalArgumentException(path + "not modifiable at runtime");
                    }
                    String[] histElement = new String[]{name, parameterName, oldValue};
                    
                    configurable.change(parameterName, defaultValue);
                    //last pushed in front
                    changedValues.add(0, histElement);
                }
                
            } catch (Exception exc) {
                //rollback is tried in reverse order
                for (String[] historyElement : changedValues) {
                    String cpName = historyElement[0];
                    String parmName = historyElement[1];
                    String histValue = historyElement[2];
                    try {
                        Configurable cp = (Configurable) lookup.getComponentByName(cpName);
                        cp.change(parmName, histValue);
                    } catch (Exception rollbackExc) {
                        throw new RollBackException(rollbackExc, cpName, parmName, histValue);
                    }
                }
                throw exc;
            }
        } )
        );
    }
    
    private AConfigProfile createConfigProfileOutOfProperties(Properties configProps, String categoryName, String configName){
        AConfigProfile res = new AConfigProfile(subsystemDescription, subsystemDescription.getUser(), PackCst.DESIGNER_LEVEL, categoryName, configName);
        if (configProps != null){
            try{
                boolean exceptionThrown = false;
                for (String name : configProps.stringPropertyNames()){
                    try{
                        res.addParameterConfiguration(name, configProps.getProperty(name));
                    } catch (Exception ex){
                        exceptionThrown =true;
                    }
                }
                if (exceptionThrown) throw new IllegalArgumentException("Some parameters were not modifiable, invoke getReConfigurationFailures") ;;
            } catch (IllegalArgumentException exc) {
                List<String> messages = res.reportFailures();
                throw new IllegalArgumentException(" configuration errors :"+ messages) ;
            }
        }
        return res;
    }
    
    public ConfigProfile getProfileForCat(String categoryName){
        return configProfileMap.get(categoryName);
    }
    

    /**
     * Configuration parameters are grouped by the name of component they belong to.
     * @param categories
     * @return 
     */
    private Map<String, List<ParameterConfiguration>> getConfigurationMapForCat(Collection<String> categories) {
        return getParameterConfigurationsForCat(categories).stream()
                .collect(Collectors.groupingBy(c -> c.getPath().getComponentName()));
    }
    
    private Set<AParameterConfiguration> getParameterConfigurationsForCat(Collection<String> categories){
        Set<AParameterConfiguration> res = new HashSet<>();
        for(String category : categories){
            res.addAll(configProfileMap.get(category).getParameterConfigurations());
        }
        return res;
    }
    
    /**
     * prepares a new executable configuration data with these modified parameters.
     * used by <TT>PreparedConfiguration</TT> objects.
     *
     * @return
     */
    public DescriptiveNode getModifiedConfigurationData() {
        DescriptiveNode res = null;
        DescriptiveNode componentNode = subsystemDescription.getTopComponentNode().clone() ;
        for (AParameterConfiguration parameterConfiguration : getParameterConfigurationsForCat(subsystemDescription.getCategorySet())) {
            ParameterPath path = parameterConfiguration.getPath();
            String componentName = path.getComponentName();
            String codeName = path.getCodeName();
            if (codeName != null && !"".equals(codeName)) {
                throw new UnsupportedOperationException(" no change on methods yet --> " + codeName);
            }
            String parameterName = path.getParameterName();
            DescriptiveNode goalComponent = (DescriptiveNode) componentNode.getNodeByName(componentName);
            if(goalComponent == null) {
                throw new IllegalArgumentException("no component for name :" + componentName) ;
            }
            Map mapAttributes = goalComponent.getAttributes() ;
            // not null
            if(mapAttributes == null ) {
                throw new  IllegalArgumentException("incompatible attribute list for component/parameter " +componentName +"/" +parameterName) ;
            }
            // get Parameter
            Object rawParm = mapAttributes.get(parameterName);
            if (rawParm instanceof HollowParm) {
                HollowParm hollow = (HollowParm) rawParm;
                hollow.modifyChecked(parameterConfiguration.getConfiguredValue());
            } else {
                throw new IllegalArgumentException("parameter not modifiable" + rawParm);
            }
        }
        res = componentNode;
        
        return res;
    }
    
    /**
     * 
     * @param configurable
     * @param parameterName
     * @param value 
     */
    @Override
    public void change(Configurable configurable, String parameterName, Object value) throws Exception {
        Method method = configurable.getEnvironment().getConfigMethodOfComponent(parameterName);
        
        ViewValue data = checkForParameterChange(configurable.getName(), parameterName, value);
        method.invoke(configurable.getEnvironment().getRelevantObject(), data.getValue());
        
        notifyParameterChange(configurable.getName(), parameterName, data.getView());
    }
    
}
