package org.lsst.ccs.config;

import java.io.Serializable;
import java.util.Collection;

/**
 * A set of static methods to be used on Configuration service client's side : they are hiding some implementation details to the "outside"
 * world.
 * To be used in conjunction with <TT>ConfigurationFacade</TT> (but
 * these codes may be called locally without referring to a server)
 * @author bamade
 */
// Date: 05/06/12

public class Factories {
    
    private Factories() {
    }
    
    ////////// creation of "raw" subsystemDescriptions
    //TODO: normally the creation of all descriptions should be made from the text file
    // so no more public factories for these ?

    /**
     * Creates a new {@code SubsystemDescription} object without any {@code ParameterDescription} list creation. 
     * The object is not registered in the database since it is to be populated by <TT>ParameterDescription</TT> objects.
     * <p/>
     * To populate the descriptions get the base descriptions from the Subsystemdescription by calling a version
     * of <TT>getBaseParameters</TT> or <TT>getParameterDescriptions</TT> and generate the descriptions from these bases.
     *
     * @param subsystemName
     * @param tag
     * @param user
     * @param version
     * @param configurationData
     * @param dataFlavour
     * @return a {@code SubsystemDescription} object without parameter descriptions
     */
    public static SubsystemDescription createRawSubsystemDescription(String subsystemName, String tag, String user, String version,
                                                              Serializable configurationData, DataFlavour dataFlavour) {
        return new ASubsystemDescription(subsystemName, tag, user, version, configurationData, dataFlavour);
    }

    /**
     * creates a new {@code SubsystemDescription} populated with "empty" {@code ParameterDescriptions}.
     * The object is not in the database since the list of descriptions should be modified first.

* @param subsystemName
     * @param tag
     * @param user
     * @param version
     * @param configurationData
     * @param dataFlavour
     * @param filter
     * @return a {@code SubsystemDescription} object containing the description of the parameters
     * as described in {@code configurationData}.
     */
    public static SubsystemDescription createSubsystemDescription(String subsystemName, String tag, String user, String version,
                                                           Serializable configurationData, DataFlavour dataFlavour, ParameterFilter filter) {
        SubsystemDescription res = createRawSubsystemDescription(subsystemName, tag, user, version, configurationData, dataFlavour);
        Collection<ParameterDescription> descriptions = res.getPossibleDescriptions(PackCst.DESIGNER_LEVEL, filter) ;
        res.addParameterDescriptions(descriptions);
        return res;
    }


    /**
     * creates a new SubsystemDescription from another. The other one could be active of deprecated data:
     * this method is meant to create another subsystemdescription were only the ParameterDescription are to be changed.
     * not that the list of ParameterDescription may be populated (if the model have some) and then that this set
     * is to be modified (the resulting object is not yet persisted in the database).
     *
     * @param desc
     * @return a deep copy of a the object.
     */
    public static SubsystemDescription createSubsystemDescriptionCopy(SubsystemDescription desc) {
        return new ASubsystemDescription(desc);
    }


    /**
     * factory method to create a new ParameterDescription. To be used by tools such as GUI
     *
     * @param parameterBase
     * @param description
     * @param simpleName
     * @param constraints
     * @param category
     * @param level
     * @return a {@code ParameterDescription} object built out of the issued
     * arguments
     */
    public static ParameterDescription createParameterDescription(ParameterBase parameterBase, String description, String simpleName, String constraints, String category, int level) {
        return new AParameterDescription(parameterBase, description, simpleName, constraints,category, level);
    }

    /**
     * Factory method to create a new ParameterDescription from another one.
     *
     * @param other
     * @return a copy of other.
     */
    public static ParameterDescription createParameterDescription(ParameterDescription other) {
        return new AParameterDescription(other);
    }

    /**
     * tries to copy <TT>ParameterDescription</TT> from a model to a new subsystem description.
     * each <TT>ParameterDescription</TT> is checked against the configData (if not compatible it is rejected
     * and a Listener is notified).
     * <p/>
     * NOT IMPLEMENTED YET
     * </P>
     *
     * @param newDescription   a subsystem description <B>without</B> any <TT>ParameterDescription</TT>
     * @param model
     * @param mismatchListener optional code that will be warned when inconsistencies occur
     */
    public static void tryCopyParameters(SubsystemDescription newDescription, SubsystemDescription model,
                                  DescriptionMismatchListener mismatchListener) {
        // generate a bogus ParameterBase list for newDescription
        // test if type is the same ?(default value may be changed)
        // new defaultValue tested against "constraints"
        // level set to old level
        System.err.println("Try copy Parameters not implemented yet");
    }


    ////////////// creation of "raw" configProfiles

    /**
     * Creates an empty ConfigProfile.
     * This object is associated to a subsystem description, is specified the 
     * category of parameters it referes to and is associated a configuration name
     * but it does not contain any {@code ParameterConfiguration} list.
     *
     * @param subsystemDesc should  be read from the database
     * @param userName
     * @param level
     * @param category
     * @param configName
     * @return an empty {@code ConfigProfile}
     * @throws IllegalArgumentException if subsystemdescription not in database
     */
    public static ConfigProfile createRawConfigProfile(SubsystemDescription subsystemDesc, String userName, int level, String category
            , String configName) {
        if (!(subsystemDesc instanceof ASubsystemDescription)) {
            throw new IllegalArgumentException("deprecated Description");
        }
        return createRawConfigProfile((ASubsystemDescription) subsystemDesc, userName, level, category, configName);
    }

    static AConfigProfile createRawConfigProfile(ASubsystemDescription subsystemDesc,  String userName, int level, String categoryName
            , String configName) {
        //TODO: check for preconditions
        return new AConfigProfile(subsystemDesc, userName, level, categoryName, configName);
    }

    /// possible parameters is a method of A configProfile

    /**
     * Creates a ConfigProfile with ParameterConfiguration where values
     * are just copied from the Description default value.
     *
     * @param subsystemDesc should be already in the database
     * @param userName
     * @param level
     * @param categoryName
     * @param configName
     * @return
     * @throws IllegalArgumentException if subsystemdescription not in database
     */
    public static ConfigProfile createConfigProfile(SubsystemDescription subsystemDesc, String userName, int level, String categoryName, String configName) {
        if (!(subsystemDesc instanceof ASubsystemDescription)) {
            throw new IllegalArgumentException("deprecated Description");
        }

        return createConfigProfile((ASubsystemDescription) subsystemDesc, userName, level, categoryName, configName);
    }

    static AConfigProfile createConfigProfile(ASubsystemDescription subsystemDesc, String userName, int level, String categoryName, String configName) {
        //TODO: check for preconditions
        AConfigProfile res = new AConfigProfile(subsystemDesc,  userName, level, categoryName, configName);
        return res;
    }

    /**
     * to be used to create a copy of configuration with different name , tag, etc.
     * This is used mostly to go back and forth in Engineering mode: for instance start
     * an Engineering mode with modification of parameters "on the fly"
     * <p/>
     * when this is used to create an "engineering mode" profile it is mandatory that the profile
     * to be copied is an active one (not a deprecated one)
     *
     * @param toBeCopied    for the moment should be an "active" ConfigProfile (not a deprecated one)
     * @param newName
     * @param newUserName
     * @param newLevel
     * @param toEngineering
     * @return a copy of the specified configuration profile with different name,
     * user name and level.
     */
    public static ConfigProfile copyProfile(ConfigProfile toBeCopied, String newName,
                                     String newUserName, int newLevel, boolean toEngineering) {
        if (toBeCopied instanceof AConfigProfile) {
            return new AConfigProfile((AConfigProfile) toBeCopied,
                    newUserName, newLevel, toEngineering, newName);
        }
        if (toEngineering) {
            throw new IllegalArgumentException("cannot create an engineering profile from a deprecated one");
        }
        throw new UnsupportedOperationException("copy with deprecated profile not yet implemented");
    }

    /**
     * Copies a configuration profile and removes parameter configurations that
     * act on static data.
     * @param toBeCopied
     * @param newName
     * @param userName
     * @param newLevel
     * @param getRidOfStaticParameters
     * @return a copy of the specified configuration profile that can be loaded
     * at runtime
     */
    public static ConfigProfile copyProfileForRegistration(ConfigProfile toBeCopied, String newName, String userName, int newLevel, boolean getRidOfStaticParameters ) {
        ConfigProfile res = copyProfile(toBeCopied,newName,userName,newLevel, false) ;
        if(getRidOfStaticParameters) {
            for (ParameterConfiguration parmConf : res.getModifiedParameters()) {
                if (parmConf.changingStaticData) {
                    res.removeParameterConfigurations(parmConf);
                }
            }
            res.setChangingStaticData(false);
        }
        return res ;
    }

    /**
     * Factory method to create a ParameterConfiguration object.
     * The object stands for the {@code description} parameter description and
     * attempts to assign a new configured value to it.
     * This factory method is to be used statically exclusively (ie not when a 
     * subsystem is running).
     *
     * @param description
     * @param value
     * @return a new configured value for the parameter depicted by {@code description}
     */
    public static ParameterConfiguration createParameterConfiguration(ParameterDescription description, String value) {
        if (!(description instanceof AParameterDescription)) {
            throw new IllegalArgumentException("deprecated description");
        }
        if(description.getId() == 0L) {
            throw new IllegalArgumentException("ParameterDescription not registered in base" + description) ;
        }
        return new AParameterConfiguration((AParameterDescription) description, value, true);
    }

    /**
     * factory method to create a ParameterConfiguration object.
     *
     * @param description
     * @return 
     */
    public static ParameterConfiguration createParameterConfiguration(ParameterDescription description) {
        if (!(description instanceof AParameterDescription)) {
            throw new IllegalArgumentException("deprecated description");
        }
        if(description.getId() == 0L) {
            throw new IllegalArgumentException("ParameterDescription not registered in base" + description) ;
        }
        return new AParameterConfiguration((AParameterDescription) description);
    }

    /**
     * tries to create a new ConfigProfile from an old one. The new one is matched to an active
     * subsystemDescription
     * <p/>
     * NOT IMPLEMENTED YET
     * </P>
     *
     * @param oldProfile
     * @param newDescription   an active subsystemDescription already registered in the databse
     * @param mismatchListener
     * @return
     */
    //TODO: implement that!
    public static ConfigProfile repair(ConfigProfile oldProfile, SubsystemDescription newDescription,
                                ConfigurationMismatchListener mismatchListener) {
        return null;
    }

    


}
