package org.lsst.ccs.config;

import org.hibernate.annotations.IndexColumn;
import org.lsst.gruth.jutils.*;
import org.lsst.gruth.utils.Tracer;
import org.lsst.gruth.nodes.ComponentFactory;

import javax.persistence.*;
import java.io.PrintWriter;
import java.io.Serializable;
import java.util.*;

/**
 * Description for a subsystem as saved in database.
 * The "subsystemName" and "tags" are keys (there is only one active subsystem with that name and tag, others
 * are in historical data).
 * <BR/>
 * These objects  contain :
 * <UL>
 *     <LI/> general informations : name, tag, user that created the description, version ...
 *     <LI/> an object (saved in database as a Lob) which contains a copy of the DescriptiveNode tree
 *     <LI/> a Set of <TT>ParameterDescriptions</TT>: objects that describe a parameter (see
 *     <TT>ParameterBase</TT> (<TT>ParameterPath</TT> + type and default value) with infos such as
 *     description, constraints, static nature ....
 *     <LI/> a list of <TT>DeploymentDescriptor</TT>a that helps identify the needed jars
 *     (todo: implement this feature)
 * </UL>
 * <P/>
 * A <TT>ConfigProfile</TT> object will reference such a description but references actual values
 * for modifiable parameters.
 */
@MappedSuperclass
public abstract class SubsystemDescription implements Serializable {

    /**
     * first time the description was valid : initially populated by the database
     * (but copies can get it from an original: it depends on the purpose of the copy )
     */
    private long startTimestamp;//generated
    /**
     * valid limit. defaults to eternity except when the object is pushed in history.
     * TODO: put in Ghost and make class immutable
     */
    private long endTimestamp = PackCst.STILL_VALID;
    /**
     * name of subsystem
     */
    private /*@NonNull*/ String subsystemName;
    /**
     * tag such as 'high wind'
     */
    private /*@NonNull*/ String tag = "";
    /**
     * name of user that created this
     */
    private /*@NonNull*/ String user;

    /**
     * version
     */
    private /*@NonNull*/ String version;
    /**
     * Big object! see DescriptionType enum ...
     * but not necessarily that big (so simply serializable Basic could be ok)
     */
    @Lob
    protected /*@NonNull*/ Serializable descriptionData;
    /**
     * <p/>
     * TODO: only object_tree in future releases (other formats can be provided to constructors)
     */
    @Enumerated(EnumType.STRING)
    protected  /*@NonNull*/ DataFlavour dataFlavour;
    /**
     * for which deployment  is this description valid?
     */
    @OneToMany(cascade = CascadeType.ALL, fetch = FetchType.EAGER)
    //@OneToMany(cascade = CascadeType.ALL)
    @IndexColumn(name = "id", base = 1)
    private /*@NonNull*/ List<DeploymentDescriptor> deployDescriptors =
            // just in case! not ideal when handled as a bean
            new ArrayList<DeploymentDescriptor>();

    /**
     * transient object to access the descriptor list
     */
    @Transient
    List<DeploymentDescriptor> unModifiableDescriptorList = Collections.unmodifiableList(deployDescriptors);

    // comments?

    /**
     * a link to the description the current object may replace
     */
    private long previousDescriptionID;

    ///////////////////////// CONSTRUCTORS


    /**
     * for bean  convention: not public!
     */
    SubsystemDescription() {
    }

    /**
     * used by subclasses
     *
     * @param subsystemName     should not be null or empty
     * @param tag               may be null or empty
     * @param user              user that "owns" the description
     * @param version           the version
     * @param descriptionData see DataFlavour documentation
     * @param dataFlavour       which type of Configuration data
     */
    protected SubsystemDescription(String subsystemName, String tag, String user, String version, Serializable descriptionData, DataFlavour dataFlavour) {
        setSubsystemName(subsystemName);
        setTag(tag);
        //TODO: precondition configuration data should not be null
        if (descriptionData == null) {
            throw new IllegalArgumentException("null configuration data");
        }
        this.user = user;
        this.version = version;
        switch (dataFlavour) {
            case TREE_FROM_SOURCE:
                if (descriptionData instanceof TextAndNode) {
                    this.descriptionData = descriptionData;
                } else {
                    this.descriptionData = new TextAndNode(descriptionData);
                }
                break;
            case DUMMY_TEXT:
            case PURE_OBJECT_TREE:
                this.descriptionData = descriptionData;
                break;
        }
        this.dataFlavour = dataFlavour;
    }
    //detach method ? copy constructor

    /**
     * creates a copy of the SubsystemDescription that is not (yet) registered to the database.
     * Subclasses <B>must</B> fill the corresponding paramDescriptions set.
     *
     * @param other
     */
    protected SubsystemDescription(SubsystemDescription other) {
        this(other.getSubsystemName(), other.getTag(), other.getUser(),
                other.getVersion(), other.getDescriptionData(), other.getDataFlavour());
        //copies the DeploymentDescriptors
        deployDescriptors.addAll(other.getDeployDescriptors());
        // danger: abstract method called in  constructor!
    }

    ///////////////////////////// ACCESSORS/MUTATORS

    /**
     * the technical id: zero if the object is not yet registered in database
     *
     * @return
     */
    public abstract long getId();

    /**
     * used only by reconstruction code
     *
     * @param id
     */
    abstract void setId(long id);

    /**
     * Detailed description of parameters that can be changed
     *
     * @return
     */
    public abstract Set<? extends ParameterDescription> getParamDescriptionSet();

    /**
     * tells if the modifying methods can be invoked on a newly created objects.
     *
     * @return false if the object has been already registered in the database
     */
    public boolean isReadOnly() {
        return getId() != 0L;
    }

    public String getSubsystemName() {
        return subsystemName;
    }

    void setSubsystemName(String subsystemName) {
        if (subsystemName == null) {
            throw new IllegalArgumentException("null subsystemName");
        }
        this.subsystemName = subsystemName;
    }

    public String getTag() {
        return tag;
    }

    void setTag(String tag) {
        if (tag == null) tag = "";
        this.tag = tag;
    }

    public long getStartTimestamp() {
        return startTimestamp;
    }

    void setStartTimestamp(long startTimestamp) {
        this.startTimestamp = startTimestamp;
    }

    public long getEndTimestamp() {
        return endTimestamp;
    }

    void setEndTimestamp(long endTimestamp) {
        this.endTimestamp = endTimestamp;
    }


    public String getUser() {
        return user;
    }

    void setUser(String user) {
        this.user = user;
    }

    public String getVersion() {
        return version;
    }

    void setVersion(String version) {
        this.version = version;
    }

    public Serializable getDescriptionData() {
        return descriptionData;
    }

    void setDescriptionData(Serializable descriptionData) {
        this.descriptionData = descriptionData;
    }

    public DataFlavour getDataFlavour() {
        return dataFlavour;
    }

    void setDataFlavour(DataFlavour dataFlavour) {
        this.dataFlavour = dataFlavour;
    }

    /**
     * get the id of the previous subsystemDescription with same Name and tag.
     * This data is modified by the configuration facade (when replacing a subsystemDescription)
     *
     * @return 0L if there is none
     */
    public long getPreviousDescriptionID() {
        return previousDescriptionID;
    }

    void setPreviousDescriptionID(long previousDescriptionID) {
        this.previousDescriptionID = previousDescriptionID;
    }

    /**
     * The list of deployment descriptors is modifiable only for objects
     * not yet registered in database. use addDeploymentDescriptor and
     * removeDeploymentDescriptors to modify the list whil the object is not registered.
     *
     * @return an unmmodifiable view of the Deployment Descriptor list
     */
    public List<DeploymentDescriptor> getDeployDescriptors() {
        // Hibernate HACK!
        if(unModifiableDescriptorList.size() != deployDescriptors.size()) {
            unModifiableDescriptorList = Collections.unmodifiableList(deployDescriptors);
        }
        return unModifiableDescriptorList;
    }

    void setDeployDescriptors(List<DeploymentDescriptor> deploymentDescriptors) {
        this.deployDescriptors = deploymentDescriptors;
        unModifiableDescriptorList = Collections.unmodifiableList(deploymentDescriptors);
    }

    /**
     * adds a list of Deployment Descriptors
     *
     * @param descriptors
     * @throws ImmutableStateException if called on an object registered on the database
     */
    public void addDeploymentDescriptors(DeploymentDescriptor... descriptors) {
        if (isReadOnly()) {
            throw new ImmutableStateException("Deployment descriptor list");
        }
        for (DeploymentDescriptor descriptor : descriptors) {
            this.deployDescriptors.add(descriptor);
        }
    }

    /**
     * removes a list of Deployment Descriptors
     *
     * @param descriptors
     * @throws ImmutableStateException if called on an object registered on the database
     */
    public void removeDeploymentDescriptors(DeploymentDescriptor... descriptors) {
        if (isReadOnly()) {
            throw new ImmutableStateException("Deployment descriptor list");
        }
        for (DeploymentDescriptor descriptor : descriptors) {
            this.deployDescriptors.remove(descriptor);
        }
    }

    /* problems with mapping on superClass
    public Set<ParameterDescription> getParamDescriptions() {
        return paramDescriptions;
    }

     void setParamDescriptions(Set<ParameterDescription> paramDescriptions) {
        this.paramDescriptions = paramDescriptions;
    }
    */

    ///////////////////////////////// IDENT METHODS


    /**
     * compares only the name and tag of subsystems not their content!
     *
     * @param
     * @return
     */
    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof SubsystemDescription)) return false;

        SubsystemDescription that = (SubsystemDescription) o;

        if (getId() != that.getId()) return false;
        if (!getSubsystemName().equals(that.getSubsystemName())) return false;
        String tag = getTag();
        if (tag != null ? !tag.equals(that.getTag()) : that.getTag() != null) return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result = getSubsystemName().hashCode();
        String tag = getTag();
        result = 31 * result + (tag != null ? tag.hashCode() : 0);
        long id = getId();
        result = 31 * result + (int) (id ^ (id >>> 32));
        return result;
    }

    @Override
    public String toString() {
        return "{" +
                "id=" + getId() +
                "; name/tag=" + subsystemName +"/"+tag +
                ";descriptions=" + this.getParamDescriptionSet() +
                '}';
    }
//////////////////////////////  METHODS


    /**
     * Any <TT>ParameterDescription</TT> can be queried using any other <TT>PathObject</TT>
     *
     * @param path
     * @return
     */
    public ParameterDescription fetch(PathObject path) {
        for (ParameterDescription description : this.getParamDescriptionSet()) {
            if (description.getPath().equals(path.getPath())) {
                return description;
            }
        }
        return null;
    }

    public ParameterDescription fetch(String pathName) {
        return fetch(ParameterPath.valueOf(pathName)) ;
    }

    /**
     * adds a list of parameter descriptions
     *
     * @param descriptions
     * @throws ImmutableStateException if called on an immutable object
     */
    public abstract void addParameterDescriptions(ParameterDescription... descriptions);

    public abstract void addParameterDescriptions(Collection<ParameterDescription> descriptions);

    /**
     * removes a list of parameter descriptions
     *
     * @param descriptions
     * @throws ImmutableStateException if called on an immutable object
     */
    public abstract void removeParameterDescriptions(ParameterDescription... descriptions);

    ////////////////////////////////////// UTILITIES

    /**
     * returns the base object structure that describes the subsystem description (as loaded from a text file).
     * BEWARE: this data does not describe precisely <TT>ParameterDescriptions</TT> that may have been edited
     * between creation and this request.
     *
     * @return
     */
    public DescriptiveNode getTopComponentNode() {
        DescriptiveNode top;
        switch (this.dataFlavour) {
            case TREE_FROM_SOURCE:
                top = ((TextAndNode) this.descriptionData).getComponentNode();
                break;
            case PURE_OBJECT_TREE:
                if (!(this.descriptionData instanceof DescriptiveNode)) {
                    throw new IllegalArgumentException("Object data not a ComponentNode");
                }
                top = (DescriptiveNode) this.descriptionData;
                break;
            default:
                throw new UnsupportedOperationException(dataFlavour + " not supported yet!");

        }
        return top;
    }


    /**
     * checks if a real subsystem can be built from the descriptionData: all objects in the
     * component node are built but the subsystem at the top is not started.
     *
     * @return a built subsystem
     * @throws Exception if anything goes wrong (classes not found, improper build, etc.)
     */
    public EffectiveNode check() throws Exception {
        DescriptiveNode top = getTopComponentNode();
        checkNodeTree(top);
        //TODO: change to CCSComponentFactory
        ComponentFactory factory = new ComponentFactory(top);
        return factory.build();
    }

    private void checkNodeTree(DescriptiveNode top) {
        // added pre-check for constraints
        Collection<ParameterDescription> parmDesc = parameterDescriptionsFromNode(top, DEFAULT_TREE_PARAMETER_FILTER, PackCst.DESIGNER_LEVEL);
        for (ParameterDescription parameterDescription : parmDesc) {
            Object realValue = Constraints.check(parameterDescription.getTypeName(), parameterDescription.getDefaultValue(), parameterDescription.getConstraints());
        }
        ArrayList<DescriptiveNode> children = top.getChildren();
        if (children != null) {
            for (DescriptiveNode child : children) {
                checkNodeTree(child);
            }
        }
    }

    /**
     * writes to a property file properties that can be edited to generate a ConfigProfile
     *
     * @param writer
     * @param levelMax
     */
    public void generateConfigProperties(PrintWriter writer, int levelMax) {
        List<ParameterDescription> list = new ArrayList<ParameterDescription>(getParamDescriptionSet());
        Collections.sort(list, PathObject.COMPARATOR);
        for (ParameterDescription description : list) {
            if (description.getLevel() <= levelMax) {
                writer.println(description.toPropertyString());
            }
        }
    }


    /**
     * to be used to create startup data object from a non-registered Configuration data (in a properties file).
     * use at your own risk!
     *
     * @param configProps
     * @return
     * @throws IllegalStateException if the set of ParameterDescription is not populated
     *                               Runtime Exception if illegal modification try (type or immutable object)
     */
    public DescriptiveNode getModifiedConfigurationData(Properties configProps) throws RuntimeException {
        DescriptiveNode res = getTopComponentNode().clone();
        Set<? extends ParameterDescription> paramDescriptions = this.getParamDescriptionSet();
        if (paramDescriptions.size() == 0) {
            throw new IllegalStateException("Description with no parametersDescription: populate first");
        }
        for (ParameterDescription parameterDescription : paramDescriptions) {
            ParameterPath path = parameterDescription.getPath();
            String pathName = path.toString();
            String value = configProps.getProperty(pathName);
            if (value == null) { //try simpleName
                value = configProps.getProperty(parameterDescription.getSimpleName());
                if (value == null) continue;
            }
            // we have a String value
            Object realValue = Constraints.check(parameterDescription.getTypeName(), value, parameterDescription.getConstraints());
            String componentName = path.getComponentName();
            String codeName = path.getCodeName();
            if (codeName != null && !"".equals(codeName)) {
                throw new UnsupportedOperationException(" no change on methods yet --> " + codeName);
            }
            String parameterName = path.getParameterName();
            DescriptiveNode goalComponent = (DescriptiveNode) res.getNodeByName(componentName);
            // get Parameter
            Object rawParm = goalComponent.getAttributes().get(parameterName);
            // not null
            if (rawParm instanceof HollowParm) {
                HollowParm hollow = (HollowParm) rawParm;
                hollow.modifyChecked(realValue);
                //System.out.println("modified :" + componentName + "//" +parameterName + " = " +realValue);
                Tracer.trace(Tracer.NODE_MODIF, "modified parameter ", componentName, "//",
                        parameterName , ". new value=", realValue);
            } else {
                throw new IllegalArgumentException("parameter not modifiable" + rawParm);
            }
        }
        //System.out.println("modified node :" +res);
        Tracer.trace(Tracer.NODE_MODIF, "modification of node ", res);
        return res;
    }


    /**
     * returns a Map that contains all ParameterDescription both by their pathName and their simpleName
     *
     * @return
     */
    public Map<String, ParameterDescription> generateDescriptionMap() {
        Map<String, ParameterDescription> map = new HashMap<String, ParameterDescription>();
        for (ParameterDescription description : getParamDescriptionSet()) {
            map.put(description.getPath().toString(), description);
            String simpleName = description.getSimpleName();
            if (simpleName != null && !"".equals(simpleName.trim())) {
                map.put(simpleName.trim(), description);
            }
        }
        return map;
    }

    /**
     * gets the base parameters from a description
     *
     * @param filter drops unwanted parameter bases
     * @return
     */
    public Collection<ParameterBase> getBaseParameters(ParameterFilter filter) {
        DescriptiveNode top = getTopComponentNode();
        return getBaseParametersFromTree(top, filter);
    }

    /**
     * Default filter for parameters: gets rid of all parameters that are references to other components
     * and considers that a parameter with name "name" is out.
     */
    public static final ParameterFilter DEFAULT_TREE_PARAMETER_FILTER = new ParameterFilter() {
        @Override
        public boolean filter(String parameterName, Object value) {
            if ("name".equalsIgnoreCase(parameterName)) return false;
            if (value instanceof NamedRefParm) return false;
            return true;
        }
    };

    /**
     * get the base parameters using the default filter.
     *
     * @return
     */
    public Collection<ParameterBase> getBaseParameters() {
        //todo DescriptiveNode
        DescriptiveNode top = getTopComponentNode();
        return getBaseParametersFromTree(top, DEFAULT_TREE_PARAMETER_FILTER);
    }

    /**
     * creates a Collection of <TT>ParameterBase</TT> objects from "tree" data that represents
     * a subsystem.
     *
     * @param filter
     * @return
     */
    Collection<ParameterBase> getBaseParametersFromTree(DescriptiveNode top, ParameterFilter filter) {
        ArrayList<ParameterBase> list = new ArrayList<ParameterBase>();
        populateParameterBasesFromTop(list, top, filter);
        return list;
    }

    /**
     * gets the <TT>ParameterBase</TT> objects that describes arguments passed to the constructor
     * linked to a node of the configuration tree.
     * <p/>
     * this code should be changed to describe also methods' parameters.
     * </P>
     *
     * @param node
     * @param filter
     * @return
     */
    public static Collection<ParameterBase> parametersFromNode(DescriptiveNode node, ParameterFilter filter) {
        Collection<ParameterBase> res = new ArrayList<ParameterBase>();
        Map attributes = node.getAttributes();
        String nodeKey = node.getKey();
        //TODO: handle positional parameters!!
        //TODO: handle other methods (not only constructors)
        if (attributes != null) {
            Set<Map.Entry> keyVals = attributes.entrySet();
            for (Map.Entry entry : keyVals) {
                Object val = entry.getValue();
                String parmName = (String) entry.getKey();
                if (!filter.filter(parmName, val)) {
                    continue;
                }
                //TODO: handle structParm
                if (val instanceof HollowParm) {
                    HollowParm hollow = (HollowParm) val;
                    if (!hollow.isReadOnly()) {
                        // first get the type
                        String typeName = hollow.getValueClass().getName();
                        ParameterBase base = new ParameterBase(nodeKey, "", parmName, typeName, hollow.toString());
                        res.add(base);
                    }
                }
            }
        }
        return res;
    }

    /**
     * "tree-walker" to populate <TT>ParameterBase</TT> collection
     *
     * @param list
     * @param topNode
     * @param filter
     */
    public static void populateParameterBasesFromTop(Collection<ParameterBase> list, DescriptiveNode topNode,
                                                     ParameterFilter filter) {
        list.addAll(parametersFromNode(topNode, filter));
        ArrayList<DescriptiveNode> children = topNode.getChildren();
        if (children != null) {
            for (DescriptiveNode childNode : children) {
                populateParameterBasesFromTop(list, childNode, filter);
            }
        }
        //TODO: handle calls !
    }

    ////////////////////////////// getting parameterDescription templates
    ///////////// TODO: this code is almost a copy of the methods for ParameterBase : mutualize!
    ////////////////////////////////////

    /**
     * Loks like the <TT>getBaseParameters</TT> but builds a collection of <TT>ParameterDescription</TT>.
     * The additional data may be empty if no additional data has been specified in the original text file/
     *
     * @param maxLevel
     * @param filter
     * @return
     */
    public Collection<ParameterDescription> getPossibleDescriptions(int maxLevel, ParameterFilter filter) {
        DescriptiveNode top = getTopComponentNode();
        ArrayList<ParameterDescription> list = new ArrayList<ParameterDescription>();
        populateParameterDescriptionsFromTop(list, top, filter, maxLevel);
        return list;
    }

    public Collection<ParameterDescription> getPossibleDescriptions(int maxLevel) {
        return getPossibleDescriptions(maxLevel, DEFAULT_TREE_PARAMETER_FILTER);
    }

    public static void populateParameterDescriptionsFromTop(Collection<ParameterDescription> list,
                                                            DescriptiveNode topNode, ParameterFilter filter, int maxLevel) {
        list.addAll(parameterDescriptionsFromNode(topNode, filter, maxLevel));
        ArrayList<DescriptiveNode> children = topNode.getChildren();
        if (children != null) {
            for (DescriptiveNode childNode : children) {
                populateParameterDescriptionsFromTop(list, childNode, filter, maxLevel);
            }
        }
        //TODO: handle calls !

    }

    public static Collection<ParameterDescription> parameterDescriptionsFromNode(DescriptiveNode node,
                                                                                 ParameterFilter filter,
                                                                                 int maxLevel) {
        Collection<ParameterDescription> res = new ArrayList<ParameterDescription>();
        Map attributes = node.getAttributes();
        String nodeKey = node.getKey();
        //TODO: handle positional parameters!!
        //TODO: handle other methods (not only constructors)
        if (attributes != null) {
            Set<Map.Entry> keyVals = attributes.entrySet();
            for (Map.Entry entry : keyVals) {
                Object val = entry.getValue();
                String parmName = (String) entry.getKey();
                if (!filter.filter(parmName, val)) {
                    continue;
                }
                //TODO: handle structParm
                if (val instanceof HollowParm) {
                    HollowParm hollow = (HollowParm) val;
                    if (!hollow.isReadOnly()) {
                        // first get the type
                        String typeName = hollow.getValueClass().getName();
                        ParameterBase base = new ParameterBase(nodeKey, "", parmName, typeName, hollow.toString());
                        String description = "";
                        String simpleName = "";
                        String constraints = "";
                        boolean notModifiableAtRuntime = false ;
                        int level = PackCst.DESIGNER_LEVEL;
                        Properties props = hollow.getProperties();
                        if (props != null) {
                            // description
                            description = props.getProperty("description", "");
                            // simpleName
                            simpleName = props.getProperty("simpleName", "");
                            // constraints
                            constraints = props.getProperty("constraints", "");
                            // level
                            String request = props.getProperty("level");
                            if (request != null) {
                                level = Integer.parseInt(request);
                            }
                            String isStatic = props.getProperty("static") ;
                            if(isStatic != null) {
                                notModifiableAtRuntime = Boolean.valueOf(isStatic) ;
                            }
                        }
                        if (level <= maxLevel) {
                            AParameterDescription parmDescription = new AParameterDescription(base, description, simpleName, constraints, level);
                            parmDescription.setNotModifiableAtRuntime(notModifiableAtRuntime);
                            res.add(parmDescription);
                        }
                    }
                }
            }
        }
        return res;
    }


}

