package org.lsst.ccs.config;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.MessagingFactory;
import org.lsst.ccs.config.utilities.ConfigUtils;
import org.lsst.ccs.framework.ConfigurationProxy;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.structs.ViewValue;
import org.lsst.gruth.types.TypeUtils;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.Optional;

/**
 * register locally changes in parameters (that is creates a new Properties "resource" for the
 * configuration)
 * <P/>
 * @author bamade
 */
// Date: 04/10/12

public class LocalConfigurationProxy implements ConfigurationProxy {

    private final static String FILE_SEPARATOR = System.getProperty("file.separator");

    /**
     * when locally registering properties the <TT>LocalConfigurationProxy</TT> may need to write
     *  properties "somewher" (be it a File or in memory for tests purposes)
     * classes that support this interface are able to create
     * the corresponding resource.
     */
    @FunctionalInterface
    public static interface WriterProvider {
        public PrintWriter getPrintWriter( NamesAndTag namesAndTag) throws IOException;

    }
    private ASubsystemDescription subsystemDescription ;
    /**
     * the reference configuration that is running in normal mode
     */
    private AConfigProfile baseProfile ;
    /**
     * the configuration that is running in engineering/normal mode
     * (in normal mode both configurations are the same)
     */
    private AConfigProfile currentProfile ;

    // previously was beenInEngineering mode (see ConfigProfile)
    private volatile boolean configurationContextStarted ;

    private Subsystem actualSubsystem ;

    // should be used for communication with the configuration database (so instead move to another
    // configuration Proxy
    private MessagingFactory fac ;

    Logger logger = Logger.getLogger("org.lsst.ccs.config");

    private WriterProvider writerProvider = new WriterProvider() {
        @Override
        public PrintWriter getPrintWriter( NamesAndTag namesAndTag) throws IOException {
            String baseName = ConfigUtils.baseNameFromNames(namesAndTag);

            //Check if the properties file already exists in the Bootstrap Environment
            String pathInBootstrap = BootstrapResourceUtils.getPathOfPropertiesFileInUserResourceDirectories(baseName);
            //If it does not exist...
            if ( pathInBootstrap == null ) {
                // then check if a there is a Bootstrap user defined directory in which to write it.
                String topMostUserDirectory = BootstrapResourceUtils.getTopUserResourceDirectory();
                //If there is not Bootstrap user Directory defined.....
                if ( topMostUserDirectory == null ) {
                    //Set the path to org.lsst.ccs.workdir
                    String workdir = BootstrapResourceUtils.getBootstrapSystemProperties().getProperty("org.lsst.ccs.workdir", "");
                    if ( !workdir.isEmpty() && ! workdir.endsWith(FILE_SEPARATOR)) {
                        workdir += FILE_SEPARATOR;
                    }
                    baseName = workdir+baseName;
                } else {
                    baseName = topMostUserDirectory+baseName;
                }
            } else {
                baseName = pathInBootstrap;
            }

            if ( ! baseName.endsWith(".properties") ) {
                baseName += ".properties";
            }

            logger.info("Saving configuration  to "+baseName);


            PrintWriter printWriter = new PrintWriter(baseName, "ISO-8859-1") ;
            return printWriter ;
        }
    } ;

    //TODO : ctors

    public LocalConfigurationProxy(ConfigProfile configProfile) {
        baseProfile = (AConfigProfile) configProfile ;
        subsystemDescription = baseProfile.getSubsystemDesc() ;
    }

    boolean isInEngineeringMode() {
        if(actualSubsystem == null) return false ;
        return actualSubsystem.isInEngineeringMode() ;
    }

    @Override
    public synchronized String getConfigurationName() {
        return currentProfile.getName() ;
    }

    @Override
    public synchronized String getTagName() {
        return currentProfile.getTag() ;
    }

    @Override
    public synchronized void startNewConfigurationContext() {
        if(! isInEngineeringMode()) {
            // todo : this.actualSubsystem.switchToEngineeringMode() ;
            // to be discussed
            throw new IllegalStateException("not in engineering mode") ;
        }
        if(configurationContextStarted) {
            // make it idempotent and return
            return ;
            //throw new IllegalStateException("already in configuraiton context") ;
        }
        configurationContextStarted = true ;
        // copy the configProfile and put it in engineering mode
        currentProfile = new AConfigProfile(baseProfile, baseProfile.getName(),
                 "", PackCst.DESIGNER_LEVEL, true) ;
        // currentProfile = voir factories
    }

    @Override
    public ViewValue checkForParameterChange(String componentName, String parameterName, Object value) {
        String strValue ;
        if(value instanceof String){
           strValue = (String)  value ;
        } else {
            strValue = TypeUtils.stringify(value) ;
        }
        // check for engineering mode and create new
        if(! configurationContextStarted) {
            startNewConfigurationContext();
        }
        ParameterPath path = new ParameterPath(componentName,"",parameterName);
        ParameterDescription parameterDescription = subsystemDescription.fetch(path) ;
        if(null == parameterDescription) {
            throw new IllegalArgumentException("incoherent parameter name for " + parameterName + "-> "
            //+ subsystemDescription.getParamDescriptions());
            + subsystemDescription.getSubsystemName());
        }
        if(parameterDescription.isNotModifiableAtRuntime()) {
            throw new IllegalStateException(" parameter " + parameterName + " not modifiable at runtime");
        }
         Object res = parameterDescription.checkValue(strValue) ;
        return new ViewValue(strValue, res) ;
    }

    @Override
    public synchronized void notifyParameterChange(String componentName, String parameterName, String value) {
        ParameterPath path = new ParameterPath(componentName,"",parameterName);
        //if static
        currentProfile.temporaryChangeConfigurationValue(path.toString(), System.currentTimeMillis(), value, true);
    }

    @Override
    public synchronized void notifyUncheckedParameterChange(String componentName, String parameterName, Object value) {
        ParameterPath path = new ParameterPath(componentName,"",parameterName);
        String strValue ;
        if( value instanceof String) {
           strValue = (String)  value ;
        } else {
            strValue = TypeUtils.stringify(value) ;
        }
        //TODO : warn !!!
        currentProfile.temporaryChangeConfigurationValue(path.toString(), System.currentTimeMillis(), strValue, false);
    }

    public synchronized  void  registerConfiguration(String configurationName) throws IOException{
        registerConfiguration(configurationName, null);
    }
    @Override
    public synchronized  void  registerConfiguration(String configurationName, String tagName) throws IOException{
        if(!configurationContextStarted) {
            //todo: throw Exception?
            return ;
        }
        if(tagName != null) {
            if( ! tagName.equals(subsystemDescription.getTag())) {
                throw new IllegalArgumentException("tag of configuration and of subsystem should be the same") ;
            }
        }
        NamesAndTag namesAndTag = new NamesAndTag(subsystemDescription.getSubsystemName(),
                configurationName,subsystemDescription.getTag()) ;
        // copy from engineering mode to normal
        baseProfile = new AConfigProfile(currentProfile, currentProfile.getName(),
                 "", PackCst.DESIGNER_LEVEL, false) ;
        currentProfile = baseProfile.clone() ;

        PrintWriter printWriter = writerProvider.getPrintWriter( namesAndTag);
        currentProfile.generateConfigProperties(printWriter);
        // make subsystem send change!
        configurationContextStarted = false ;
        printWriter.flush() ;
        printWriter.close() ;
    }


    @Override
    public synchronized void dropModifications() {
        currentProfile =  baseProfile.clone() ;
        // make subsystem change notification
        configurationContextStarted = false ;
    }

    @Override
    public Object getDefaultParameterValue(String componentName, String parameterName) {
        ParameterPath path = new ParameterPath(componentName,"",parameterName);
        ParameterDescription parameterDescription = subsystemDescription.fetch(path) ;

        String strValue = parameterDescription.getDefaultValue();
        String type = parameterDescription.getTypeName() ;
        Object res = Constraints.check(type,strValue, null) ;
        return res;
    }

    @Override
    public void setSubsystem(Subsystem subsystem) {
        this.actualSubsystem= subsystem ;
        this.fac = subsystem.getMessagingAccess() ;
    }

    @Override
    public Optional<Subsystem> getSubsystem() {
        return Optional.ofNullable(actualSubsystem);
    }

    public AConfigProfile getBaseProfile() {
        return baseProfile;
    }

    public void setWriterProvider(WriterProvider writerProvider) {
        this.writerProvider = writerProvider;
    }
}
