package org.lsst.ccs.config;

import org.lsst.gruth.jutils.DescriptiveNode;

import javax.persistence.MappedSuperclass;
import java.io.PrintWriter;
import java.io.Serializable;
import java.util.*;

/**
 * This class represents a set of parameter that have been modified for a given subsystem.
* <P>
 *     <B>Important notice</B> : for the moment an active configuration profile has a name and a tag.
 *     This combination is unique amongst active profile. BUT this is not the case once data of this type
 *     has been "pushed in history" :
 *     <UL>
 *         <LI> in history there are many versions of the same config profile
 *         <LI> it is not guaranteed that the same config name may be re-used for another subsystem
 *     </UL>
 *     So these naming strategies should be re-evaluated after some experimental uses.
 * @author bamade
 */
// Date: 11/04/12
@MappedSuperclass
public abstract class ConfigProfile implements Serializable {
    /**
     * first time the profile was valid
     */
    private long startTimestamp ;//generated
    /**
     * valid limit. defaults to eternity except when the object is pushed in history.
     */
    private long endTimestamp = PackCst.STILL_VALID ;

    /**
     *
     */
    private String name ;

    /**
     *
     */
    private String tag ;

    /**
     * the user
     */
    private String userName ;

    //TODO add version

    /**
     * this may be used to register a ConfigProfile which is in the state of being modified
     * in the context of Engineering mode. it is used but not yet registered and tagged.
     * ParameterConfiguration may have an history of modifications.
     *
     */
    // TODO engineering mode use?
    // Todo: suppress ? inital intent was to notify that this was created in Engineering : another name is used in modification see ConfigurationProxy
    private boolean beenInEngineeringMode ;


    /**
     * the level (defaults to END USER)
     * see PackCst.
     */
    //TODO: normalize with Command levels
    private int level = PackCst.END_USER_LEVEL ;

    /**
     * comments
     */
    private String remarks ;

    /**
     * config with same name and tag: id of previous version
     */
    private long previousConfigID ;

    ///////////////////////////// CONSTRUCTORS


    protected ConfigProfile() {
    }

    protected ConfigProfile(String name, String tag, String userName, int level) {
        this.name = name ;
        this.tag = tag ;
        this.userName = userName;
        this.level = level;
    }
    //TODO :copy constructor

    /////////////////////////// ACCESSORS/MUTATORS
    protected abstract long getId() ;
    protected abstract void setId(long id) ;

    public abstract SubsystemDescription getSubsystemDescription() ;

    public abstract Set<? extends ParameterConfiguration> getModifiedParameters() ;

    public abstract void temporaryChangeConfigurationValue(ParameterConfiguration parameter, long time, String value, boolean checked) ;

    public  abstract ParameterConfiguration temporaryChangeConfigurationValue(String  parameterPath, long time, String value, boolean checked)  ;
        /*
        // build path obejct
        ParameterPath path = ParameterPath.valueOf(parameterPath) ;
        // fetch the ParameterConfiguration
        ParameterConfiguration parameter = this.fetch(path) ;
        //TODO if null create a new one and put it in engineering mode
        if(parameter == null) return null ;
        //calls the other method
        temporaryChangeConfigurationValue(parameter, time, value);
        return parameter ;
    }
    */


    public boolean isReadOnly() {
        return getId() != 0L ;
    }

    public boolean isBeenInEngineeringMode() {
        return beenInEngineeringMode;
    }

    void setBeenInEngineeringMode(boolean beenInEngineeringMode) {
        this.beenInEngineeringMode = beenInEngineeringMode;
    }

    public String getName() {
        return name;
    }

    void setName(String name) {
        this.name = name;
    }


    public String getSubsystemName() {
        return getSubsystemDescription().getSubsystemName() ;
    }

    public String getTag() {
        return tag;
    }

    void setTag(String tag) {
        this.tag = tag;
    }


    public long getStartTimestamp() {
        return startTimestamp;
    }

    protected void setStartTimestamp(long startTimestamp) {
        this.startTimestamp = startTimestamp;
    }

    public long getEndTimestamp() {
        return endTimestamp;
    }

    void setEndTimestamp(long endTimestamp) {
        this.endTimestamp = endTimestamp;
    }

    public String getUserName() {
        return userName;
    }

    void setUserName(String userName) {
        this.userName = userName;
    }

    public int getLevel() {
        return level;
    }

     void setLevel(int level) {
        this.level = level;
    }

    public String getRemarks() {
        return remarks;
    }

    /**
     * no influence on behaviour.
     * @param remarks
     */
    public  void setRemarks(String remarks) {
        this.remarks = remarks;
    }

    /**
     * may return the id of a previous ConfigProfile (0L if none)
     * @return
     */
    public long getPreviousConfigID() {
        return previousConfigID;
    }

    void setPreviousConfigID(long previousConfigID) {
        this.previousConfigID = previousConfigID;
    }

    //////////////////////////// IDENT METHODS

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof ConfigProfile)) return false;

        ConfigProfile that = (ConfigProfile) o;
        if (getId() != that.getId()) return false;
        if (!getName().equals(that.getName())) return false;
        String tag = getTag() ;
        if (tag != null ? !tag.equals(that.getTag()) : that.getTag() != null) return false;

        return true;
    }

    @Override
    public int hashCode() {
        long id = getId() ;
        int result = (int) (id ^ (id >>> 32));
        result = 31 * result + getName().hashCode();
        String tag = getTag() ;
        result = 31 * result + (tag != null ? tag.hashCode() : 0);
        return result;
    }
    @Override
    public String toString() {
        return "{" +
                "id=" + getId() +
                ";configurations=" + this.getModifiedParameters() +
                '}';
    }

    /////////////////////////// OTHER METHODS
    public ParameterConfiguration fetch(PathObject path) {
        for(ParameterConfiguration config: this.getModifiedParameters()) {
            if(config.getPath().equals(path.getPath())) {
                return config ;
            }
        }
        return null ;
    }

    /**
     * prints to a property file a template that represents this ConfigProfile
     * @param printer
     */
    public void generateConfigProperties(PrintWriter printer) {
        List<ParameterConfiguration> list = new ArrayList<ParameterConfiguration>(this.getModifiedParameters());
        Collections.sort(list, PathObject.COMPARATOR);
        for(ParameterConfiguration config: list) {
            boolean commentOut = false ;
            if(config.getValue().equals(config.getDescription().getDefaultValue())) {
                commentOut=  true ;
            }
            printer.println(config.getDescription().toPropertyString(config.getValue(), commentOut));
        }
    }

    /**
     * populates this configProfile with values extracted from a Properties object.
     * @param props
     * @throws IllegalArgumentException if this profile is read only or if a value is illegal
     *
     */
    public abstract  void mergeProperties(Properties props) ;

    /**
     * registers a list of parameter configurations
     * @param parameterConfigurations
     * @throws ImmutableStateException if the object is already registered in database
     * @throws IllegalArgumentException if levels are incompatibles
     * @throws IllegalArgumentException if the corresponding parameter descriptions are not alive in the current
     * subsystem description
     */
    // TODO: check if Description not deprecated!
    public abstract void    addParameterConfigurations(ParameterConfiguration... parameterConfigurations) ;

    /**
     * removes a list of parameter configurations
     * @param parameterConfigurations
     * @throws ImmutableStateException if operated on an object registered to the database
     */
    public abstract void removeParameterConfigurations(ParameterConfiguration... parameterConfigurations) ;

    /**
     * prepares a new executable configuration data with these modified parameters.
     * used by <TT>PreparedConfiguration</TT> objects.
     * @return
     */
    public abstract DescriptiveNode getModifiedConfigurationData() ;

    /**
     * gives the value of a given parameter at a given date on a ConfigProfile object.
     * beware : the main goal of this method is to fetch a transient parameter set during
     * an engineering mode session. Otherwise it just returns the value of the parameter
     * without any time stamp check (it is supposed that the <TT>ConfigProfile</TT> object
     * was the one active at this date.
     * <P>
     * @param parameterPath
     * @param date
     * @return
     */
    public  String getValueAt( String parameterPath, long date) {
        //???? if (this == null) return null;
        if (parameterPath == null) return null;
        ParameterPath path = ParameterPath.valueOf(parameterPath);
        ParameterConfiguration parmConfig = this.fetch(path);
        if (parmConfig == null) { // look if subsystem parameter default value
            SubsystemDescription subsystemDescription = this.getSubsystemDescription();
            ParameterDescription description = subsystemDescription.fetch(path);
            if (description != null) {
                return description.getParameterBase().getDefaultValue();
            }
            return null;
        }
        if (this.isBeenInEngineeringMode()) {
            List<? extends ValueEvent> eventList = parmConfig.getValueEvents();
            if (eventList == null) {
                return parmConfig.value;
            }
            //TODO: what to do with this values?
            long begin = this.getStartTimestamp();
            long end = this.getEndTimestamp();
            String lastValue = parmConfig.value;
            for (ValueEvent event : eventList) {
                if (event.getTime() > date) {
                    return lastValue;
                }
                lastValue = event.getValue();
            }
            return null; // should NOT HAPPEN TODO: put assertion here
        } else {
            return parmConfig.value;
        }
    }



}
