package org.lsst.ccs.config;

import org.lsst.ccs.bus.MessagingFactory;
import org.lsst.ccs.config.utilities.ConfigUtils;
import org.lsst.ccs.framework.ConfigurationProxy;
import org.lsst.ccs.utilities.structs.ViewValue;
import org.lsst.gruth.types.TypeUtils;

import java.io.IOException;
import java.io.PrintWriter;
import org.lsst.ccs.bootstrap.BootstrapUtils;
import org.lsst.ccs.bootstrap.resources.BootstrapResourceUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * @author bamade
 */
// Date: 04/10/12

public class LocalConfigurationProxy implements ConfigurationProxy {
    private ASubsystemDescription subsystemDescription ;
    /**
     * the reference configuration that is running in normal mode
     */
    private AConfigProfile baseProfile ;
    /**
     * the configuration that is running in engineering/normal mode
     * (in normal mode both configurations are the same)
     */
    private AConfigProfile currentProfile ;

    private volatile boolean inEngineeringMode ;

    private MessagingFactory fac ;

    Logger logger = Logger.getLogger("org.lsst.ccs.config");

    //TODO : ctors

    public LocalConfigurationProxy(ConfigProfile configProfile) {
        baseProfile = (AConfigProfile) configProfile ;
        subsystemDescription = baseProfile.getSubsystemDesc() ;
    }

    @Override
    public synchronized String getConfigurationName() {
        return currentProfile.getName() ;
    }

    @Override
    public synchronized String getTagName() {
        return currentProfile.getTag() ;
    }

    @Override
    public synchronized void startNewConfigurationContext() {
        if(inEngineeringMode) {
            throw new IllegalStateException("already in engineering mode") ;
        }
        inEngineeringMode = true ;
        // copy the configProfile and put it in engineering mode
        currentProfile = new AConfigProfile(baseProfile, baseProfile.getName(),
                baseProfile.getTag(), "", PackCst.DESIGNER_LEVEL, true) ;
        // currentProfile = voir factories
    }

    @Override
    public ViewValue checkForParameterChange(String componentName, String parameterName, Object value) {
        String strValue ;
        if(value instanceof String){
           strValue = (String)  value ;
        } else {
            strValue = TypeUtils.stringify(value) ;
        }
        // check for engineering mode and create new
        if(! inEngineeringMode) {
            startNewConfigurationContext();
        }
        ParameterPath path = new ParameterPath(componentName,"",parameterName);
        ParameterDescription parameterDescription = subsystemDescription.fetch(path) ;
        if(null == parameterDescription) {
            throw new IllegalArgumentException("incoherent parameter name for " + parameterName + "-> "
            //+ subsystemDescription.getParamDescriptions());
            + subsystemDescription.getSubsystemName());
        }
        if(parameterDescription.isNotModifiableAtRuntime()) {
            throw new IllegalStateException(" parameter " + parameterName + " not modifiable at runtime");
        }
         Object res = parameterDescription.checkValue(strValue) ;
        return new ViewValue(strValue, res) ;
    }

    @Override
    public synchronized void notifyParameterChange(String componentName, String parameterName, String value) {
        ParameterPath path = new ParameterPath(componentName,"",parameterName);
        //if static
        currentProfile.temporaryChangeConfigurationValue(path.toString(), System.currentTimeMillis(), value);
    }

    @Override
    public synchronized void notifyUncheckedParameterChange(String componentName, String parameterName, Object value) {
        ParameterPath path = new ParameterPath(componentName,"",parameterName);
        String strValue ;
        if( value instanceof String) {
           strValue = (String)  value ;
        } else {
            strValue = TypeUtils.stringify(value) ;
        }
        //TODO : warn !!!
        currentProfile.temporaryChangeConfigurationValue(path.toString(), System.currentTimeMillis(), strValue);
    }

    @Override
    public synchronized  void  registerConfiguration(String configurationName, String tagName) throws IOException{
        if(!inEngineeringMode) {
            return ;
        }
        NamesAndTag namesAndTag = new NamesAndTag(subsystemDescription.getSubsystemName(),
                configurationName,tagName) ;
        // copy from engineering mode to normal
        baseProfile = new AConfigProfile(currentProfile, currentProfile.getName(),
                currentProfile.getTag(), "", PackCst.DESIGNER_LEVEL, false) ;
        currentProfile = baseProfile.clone() ;
        String baseName = ConfigUtils.baseNameFromNames(namesAndTag);

        //Check if the properties file already exists in the Bootstrap Environment
        String pathInBootstrap = BootstrapResourceUtils.getPathOfPropertiesFileInUserResourceDirectories(baseName);
        //If it does not exist...
        if ( pathInBootstrap == null ) {
            // then check if a there is a Bootstrap user defined directory in which to write it.
            String topMostUserDirectory = BootstrapResourceUtils.getTopUserResourceDirectory();
            //If there is not Bootstrap user Directory defined.....
            if ( topMostUserDirectory == null ) {
                //Set the path to org.lsst.ccs.workdir
                String workdir = BootstrapResourceUtils.getBootstrapSystemProperties().getProperty("org.lsst.ccs.workdir", "");
                if ( !workdir.isEmpty() && ! workdir.endsWith(BootstrapUtils.FILE_SEPARATOR)) {
                    workdir += BootstrapUtils.FILE_SEPARATOR;
                }
                baseName = workdir+baseName;
            } else {
                baseName = topMostUserDirectory+baseName;
            }
        } else {
            baseName = pathInBootstrap;
        }
        
        logger.info("Saving configuration tag "+tagName+" to "+baseName+".properties");
        
        
        PrintWriter printWriter = new PrintWriter(baseName+".properties", "UTF-8") ;
        currentProfile.generateConfigProperties(printWriter);
        // make subsystem send change!
        inEngineeringMode = false ;
        printWriter.flush() ;
        printWriter.close() ;
    }

    @Override
    public synchronized void dropModifications() {
        currentProfile =  baseProfile.clone() ;
        // make subsystem change notification
        inEngineeringMode = false ;
    }

    @Override
    public Object getDefaultParameterValue(String componentName, String parameterName) {
        ParameterPath path = new ParameterPath(componentName,"",parameterName);
        ParameterDescription parameterDescription = subsystemDescription.fetch(path) ;

        String strValue = parameterDescription.getDefaultValue();
        String type = parameterDescription.getTypeName() ;
        Object res = Constraints.check(type,strValue, null) ;
        return res;
    }

    @Override
    public void setMessagingFactory(MessagingFactory factory) {
        this.fac = factory ;
    }
}
