package org.lsst.ccs.config;

import javax.persistence.*;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

/**
 * a "zombie" that represents subsystem description that can be alive (end timestamp is STILL_ALIVE)
 * or dead (end timestamp is a realistic date).
 * <P>
 *  "ghost" data is necessary because there can be references to subsystem description
 *  that becomes deprecated while the description is "alive"; when later this description "dies"
 *  it's not necessary to change the references to this data (only the end time stamp is changed).
 *  <P>
 *   so ghosts are created at the same time as the original "living" data and anticipate death;
 *   their id is the same as the "living" object.
 * @author bamade
 */
// Date: 10/04/12

    @Entity
    // not immutable since endtime can be modified!
    //@Table(name="SubsystemHistory")
class GhostSubsystemDescription  extends SubsystemDescription{
    private static final long serialVersionUID = -8567707918287105248L;
    @Id
    private long id ;


    // here the combination "name + tag" is not unique so is not a naturalId
    // TODO: a map<Path, XXX> instead
    // IMMUTABLE!
    @OneToMany(cascade = CascadeType.ALL, fetch= FetchType.EAGER)
    protected /*@NonNull*/ Set<GhostParameterDescription> paramDescriptions  ;

    //////////// CONSTRUCTORS

     GhostSubsystemDescription() {
    }

    /**
     * creates a ghost from a living subsystem description.
     * @param other
     * @throws IllegalArgumentException if argument is not registered in database
     */
    public GhostSubsystemDescription(ASubsystemDescription other) {
        super(other);
        this.id = other.getId();
        //post control
        if(this.id == 0 ){
            throw new IllegalArgumentException(
                    "cannot create ghost from non registered object (where id is not initialized by database)");
        }
        this.setStartTimestamp(other.getStartTimestamp());
        Set<GhostParameterDescription> set = new HashSet<GhostParameterDescription>();
        // copy the ParameterDescription
        for(ParameterDescription parmDesc: other.getParamDescriptions()) {
            AParameterDescription realDesc = (AParameterDescription) parmDesc ;
            set.add(new GhostParameterDescription(realDesc)) ;
        }
        this.paramDescriptions = Collections.unmodifiableSet(set) ;
        this.setPreviousDescriptionID(other.getPreviousDescriptionID());
    }


    /////////////////////// ACCESSORS/ MUTATORS

    @Override
    public long getId() {
        return id;
    }

    @Override
    void setId(long id) {
       this.id = id ;
    }

    /**
     * calls <TT>getParamDescriptions()</TT>
     * @return
     */
    @Override
    public Set<? extends ParameterDescription> getParamDescriptionSet() {
        return getParamDescriptions() ;
    }


    /**
     *
     * @return an immutable list of ghosts for Parameter descriptions
     */
    public Set<GhostParameterDescription> getParamDescriptions() {
        return paramDescriptions;
    }

     void setParamDescriptions(Set<GhostParameterDescription> paramDescriptions) {
        this.paramDescriptions = Collections.unmodifiableSet(paramDescriptions);
    }

    @Override
     void setEndTimestamp(long endTimestamp) {
        super.setEndTimestamp(endTimestamp);
    }

    @Override
    public void addParameterDescriptions(ParameterDescription... descriptions) {
        throw new UnsupportedOperationException("immutable description") ;
    }

    @Override
    public void addParameterDescriptions(Collection<ParameterDescription> descriptions) {
        throw new UnsupportedOperationException("immutable description") ;
    }

    @Override
    public void removeParameterDescriptions(ParameterDescription... descriptions) {
        throw new UnsupportedOperationException("immutable description") ;
    }


}
