package org.lsst.ccs.config;

import org.hibernate.annotations.Immutable;

import javax.persistence.*;
import java.io.Serializable;
import java.util.*;

/**
 * An active subsystem Description.
 * Can be "unregistered" (id is zero) or "registered" in the database (Id is not zero)
 * in that case the object is immutable (more precisely only comments can be modified).
 *
 * Subsystem name and tag constitute a unique key in the active  table.
 *
 * @author bamade
 */
// Date: 10/04/12
@Table(name="ASubsystemDescription",
        uniqueConstraints = {@UniqueConstraint(columnNames={"subsystemName", "tag"})}
)
    @Entity
    @Immutable
class ASubsystemDescription  extends SubsystemDescription implements Cloneable{
    private static final long serialVersionUID = 7518445583831244327L;
    @Id
    @GeneratedValue
    private long id ; //generated
    // TODO: a map<Path, XXX> instead
    /**
     * the set containing the parameter descriptions
     */
    @OneToMany(cascade = CascadeType.ALL, fetch=FetchType.EAGER)
    protected /*@NonNull*/ Set<AParameterDescription> paramDescriptions =
            // just in case! not ideal when handled as a bean
            new HashSet<AParameterDescription>();
    /**
     * a unModifiable view of the parameter description set.
     */
    @Transient
    private Set<AParameterDescription> unModifiableParams = Collections.unmodifiableSet(paramDescriptions) ;


    /////////////////////////////// CONSTRUCTORS

    /***
     * for reconstruction tool purposes only
     */
    ASubsystemDescription(){
    }

    /**
     * same parameters and constraints as the superclass constructor
     * @param subsystemName
     * @param tag
     * @param user
     * @param version
     * @param configurationData
     * @param dataFlavour
     */
    public ASubsystemDescription(String subsystemName, String tag, String user, String version, Serializable configurationData, DataFlavour dataFlavour) {
        super(subsystemName, tag, user, version, configurationData, dataFlavour);
    }

    /**
     * creates a new SubsystemDescription without information set up by the database (id, startDate)
     * @param other
     */
    public ASubsystemDescription(SubsystemDescription other) {
        super(other) ;
        // now fill
        for(ParameterDescription parmDesc: other.getParamDescriptionSet()) {
            this.paramDescriptions.add(new AParameterDescription(parmDesc)) ;
        }
    }


    ////////////////////////////// ACCESSORS/MUTATORS

    //    @Override
    public long getId() {
        return id;
    }

    @Override
    void setId(long id) {
        this.id = id ;
    }

    @Override
    public Set<? extends ParameterDescription> getParamDescriptionSet() {
        return getParamDescriptions() ;
    }


    //@Generated(value = GenerationTime.INSERT)
    //@Temporal(TemporalType.DATE)
    @Override
    public long getStartTimestamp() {
        return super.getStartTimestamp();
    }


    /**
     *a data in this set is modifiable only if the object is not registered.
     * Use methods <TT>addParameterDescriptions</TT> and <TT>removeParameterDescriptions</TT>
     * to modify the set.
     * @return  an unmodifiable set
     */
    public Set<AParameterDescription> getParamDescriptions() {
        return unModifiableParams ;
    }

     void setParamDescriptions(Set<AParameterDescription> paramDescriptions) {
        this.paramDescriptions = paramDescriptions;
         unModifiableParams = Collections.unmodifiableSet(paramDescriptions) ;
    }

    /**
     * clone: the contained set of parameter description is also cloned.
     * (each description is cloned)
     * @return
     */
    public ASubsystemDescription clone() {
        ASubsystemDescription res = null ;
        try {
            res = (ASubsystemDescription) super.clone() ;
            HashSet<AParameterDescription> newSet = new HashSet<AParameterDescription>() ;
            for(AParameterDescription description : paramDescriptions) {
                newSet.add(description.clone()) ;
            }
            res.setParamDescriptions(newSet);
            //TODO: clone other members?
        } catch (CloneNotSupportedException e) {/*IGNORE*/ }
        return res ;
    }


//////////////////////////////
    ///////////////////// UTILITIES

    /**
     * add Parameter descriptions: this is a public facade to the "real" method
     * that adds only objects of type <TT>AParameterDescription</TT>
     * So if objects are not ot this actual type new objects of the needed type are created
     * (see <TT>AParameterDescription</TT> constructor documentation)
     * @param descriptions
     */
    @Override
    public void addParameterDescriptions(ParameterDescription... descriptions) {
        addParameterDescriptions(buildSafeArray(descriptions));
    }

    @Override
    public void addParameterDescriptions(Collection<ParameterDescription> descriptions) {
        for(ParameterDescription parm : descriptions) {
            if(! (parm instanceof AParameterDescription)) {
                this.addParameterDescriptions(new AParameterDescription(parm));
            } else {
                this.addParameterDescriptions((AParameterDescription) parm);
            }
        }
    }

    private AParameterDescription[] buildSafeArray(ParameterDescription... descriptions){
        List<ParameterDescription> list = Arrays.asList(descriptions);
        AParameterDescription[] parms = new AParameterDescription[descriptions.length] ;
        int ix = 0 ;
        for(ParameterDescription description: list) {
            if(! (description instanceof AParameterDescription)){
                parms[ix] = new AParameterDescription(description)  ;

            } else {
                parms[ix] = (AParameterDescription) description ;
            }
            ix++ ;
        }
        return parms ;
    }

    /*
    //works as is dues to equality rules
    @Override
    public void removeParameterDescriptions(ParameterDescription... descriptions) {
        removeParameterDescriptions(buildSafeArray(descriptions));
    }
    */

    /**
     * adds a list of parameter descriptions
     * @param descriptions
     * @throws ImmutableStateException if called on an immutable object
     */
    public void addParameterDescriptions(AParameterDescription... descriptions) {
        if(isReadOnly()) {
            throw new ImmutableStateException("parameter description list") ;
        }
        for(AParameterDescription description : descriptions) {
            this.paramDescriptions.add(description) ;
        }
    }
    /**
     * removes a list of parameter descriptions.
     * <P>
     *     works with any subclass of parameterDescription (see equals method code)
     * </P>
     * @param descriptions
     * @throws ImmutableStateException if called on an immutable object
     */
    public void removeParameterDescriptions(ParameterDescription... descriptions) {
        if(isReadOnly()) {
            throw new ImmutableStateException("parameter description list") ;
        }
        for(ParameterDescription description : descriptions) {
            this.paramDescriptions.remove(description) ;
        }
    }

}
