package org.lsst.ccs.commons.annotations;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * This field annotation allows to access other components from a given
 * component that belongs to the component tree. Given a tree travel
 * {@link Strategy}, it allows to collect one or more traveled components,
 * depending on the type of the annotated field. If the annotated field is an
 * array or a Collection (supported collections are
 * {@code Map<String, T>, List<T>, Set<T>, T[]}), traveled components are
 * collected only if they can be assigned to the elementary field type, until
 * the travel ends. For arrays and collection, components are inserted in
 * pre-order (parents first, children then). In case of maps, the pre-order is
 * preserved only if the map implementation preserves insertion order. If the
 * annotated field is not of a compound type, the first component met during the
 * tree travel that is assignable to the field type is set to the annotated
 * field.
 *
 * @author LSST CCS Team
 */
@Target(ElementType.FIELD)
@Retention(RetentionPolicy.RUNTIME)
@Documented
public @interface LookupField {

    /**
     * Defines the tree travel strategy.
     */
    public enum Strategy {

        /**
         * Travels from parent to parent.
         */
        ANCESTORS,
        /**
         * Travels all the descendants components.
         */
        DESCENDANTS,
        /**
         * Travels the direct children.
         */
        CHILDREN,
        /**
         * Travels the siblings.
         */
        SIBLINGS,
        /**
         * Sets the annotated field to the top component of the component tree,
         * which is the enclosing subsystem.
         */
        TOP,
        /**
         * Travels the whole component tree.
         */
        TREE,
        /**
         * Sets the annotated field to the component named after the field name.
         * @deprecated it should not be used.
         */
        @Deprecated
        BYNAME
    }

    /**
     * The travel strategy.
     *
     * @return the travel strategy
     */
    Strategy strategy();

    /**
     * Boolean argument used when filling the keys in Maps. By default maps are 
     * filled with the component name as key and the component object as value.
     * If usePath is set to true, then the path will be used as key.
     * 
     * @return true/false. True to specify that paths rather than names are to be
     *         used when filling Maps.
     */
    boolean usePath() default false;

}
