package org.lsst.ccs.common.devices.power.distribution;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.Agent;
import org.lsst.ccs.StateChangeListener;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.common.devices.power.distribution.state.PduOutletState;
import org.lsst.ccs.common.devices.power.distribution.state.PduState;
import org.lsst.ccs.common.devices.power.distribution.state.PduProperties;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.description.ComponentNode;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.apcpdu.APC7900;
import org.lsst.ccs.drivers.apcpdu.APC7900B;
import org.lsst.ccs.drivers.apcpdu.APC7900Series;
import org.lsst.ccs.drivers.apcpdu.APC7900Sim;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  APC7900 device class for the teststand subsystem
 *
 *  @author: Owen Saxton
 */
public class APC7900Device extends Device implements StateChangeListener {

    public final static int
        TYPE_OUTLET = 0,
        TYPE_POWER = 1,
        CHAN_CURRENT = 0,
        CHAN_POWER = 1;

    final static Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("OUTLET", TYPE_OUTLET);
        typeMap.put("POWER", TYPE_POWER);
    }

    @ConfigurationParameter(isFinal = true)
    private String type = "APC7900";

    @ConfigurationParameter(isFinal = true)
    protected String node;

    @ConfigurationParameter(isFinal = true)
    protected List<String> controlledOutlets = new ArrayList<>();

    protected Map<String,PduOutlet> mapOfOutlets = new HashMap<>();
    
    String user = "apc";
    String passwd = "apc";

    protected final Logger sLog = Logger.getLogger(getClass().getPackage().getName());
    private APC7900Series pdu;
    
    protected final List<String> outlets = new ArrayList<>();
    protected final List<String> outletNames = new ArrayList<>();
    protected int[] outNums;
    protected Map<String, Integer> outNumMap = new HashMap<>();

    @LookupField(strategy = LookupField.Strategy.TREE)
    private Agent a;
    
    private ComponentLookup componentLookup;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    protected AgentStateService stateService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    protected AlertService alertService;

    @LookupPath
    private String path;
    
    private boolean readPduOutlets = false;
    
    @Override
    public void build() {
        super.build(); //To change body of generated methods, choose Tools | Templates.
        
        if ( controlledOutlets.size() > 0 ) {
            readPduOutlets = true;
            componentLookup = a.getComponentLookup();
            ComponentNode thisNode = componentLookup.getComponentNodeForObject(this);
            for ( String outletStr : controlledOutlets ) {
                PduOutlet outlet = new PduOutlet(outletStr);
                mapOfOutlets.put(outlet.getName(),outlet);
                //Add the Pdu Outlet node and register its state.
                componentLookup.addComponentNodeToLookup(thisNode, new ComponentNode(thisNode, outlet.getName(), outlet));
                outletNames.add(outlet.getName());
            }
            
        }
        
        a.setAgentProperty(PduProperties.HAS_PDU, "true");
        String components = s.getAgentProperty(PduProperties.PDU_COMPONENTS);
        if (components == null) {
            components = "";
        }
        if ( !components.isEmpty() ) {
            components += ",";
        }
        components += path;
        s.setAgentProperty(PduProperties.PDU_COMPONENTS, components);
        
    }

    @Override
    public void init() {
        
        if ( type == "APC7900" ) {
            pdu = new APC7900();
        } else if ( type == "APC7900B") {
            pdu = new APC7900B();            
        } else if ( type == "APC7900Sim") {
            pdu = new APC7900Sim(outletNames);
        }
            
        
        
        //Register the Pdu State for this component
        stateService.registerState(PduState.class, "The state of the Pdu", this);        
        //And set its initial state
        stateService.updateAgentComponentState(this, PduState.NOTCONFIGURED);
                
        for ( PduOutlet outlet : mapOfOutlets.values() ) {
            stateService.registerState(PduOutletState.class, "The state of a Pdu Outlet: ON/OFF", outlet);            
        }
        
        //Add state change listener to raise alerts if/when outlets are turned off.
        stateService.addStateChangeListener(this, PduOutletState.class, PduState.class);

        super.init();
    }
    
    
    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice()
    {
        if (node == null) {
            MonitorLogUtils.reportConfigError(sLog, name, "node", "is not specified");
        }
        fullName = "APC7900 PDU (" + node + ")";
    }


    /**
     *  Initializes the device.
     */
    @Override
    protected void initialize()
    {
        try {
            pdu.open(APC7900.ConnType.TELNET, node, user, passwd);
            outNumMap = pdu.getOutletNumberMap();
            outNums = new int[outlets.size()];

            if ( controlledOutlets.isEmpty() ) {
                for ( String outletName : outNumMap.keySet() ) {
                    int index = outNumMap.get(outletName);
                    PduOutlet outlet = new PduOutlet(name,index);
                    mapOfOutlets.put(outletName, outlet);
                }
            }            
            
            if ( mapOfOutlets.size() > 0 ) {
                for ( String outlet : outNumMap.keySet() ) {
                    String outletName = outlet;
                    int outletIndex = outNumMap.get(outlet);
                    PduOutlet pduOutlet = mapOfOutlets.get(outletName);
                    if ( pduOutlet == null ) {
                        sLog.warning("PDU outlet "+outletName+" is not monitored by this subsystem.");
                    } else if ( pduOutlet.getIndex() != outletIndex ) {
                        sLog.warning("PDU outlet "+outletName+" is misconfigured. Should have inxed "+outletIndex+" rather than "+pduOutlet.getIndex());
                    }
                    sLog.warning("Monitoring outlet "+pduOutlet);
                }
            }
            
            initSensors();
            setOnline(true);
            stateService.updateAgentComponentState(this, PduState.OK);
            sLog.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                sLog.error("Error connecting to " + fullName + ": " + e);
            }
            close();
        }
        inited = true;
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            pdu.close();
        }
        catch (DriverException e) {
        }
        outNumMap.clear();
        stateService.updateAgentComponentState(this, PduState.NOTCONFIGURED);
    }

    @Override
    public void stateChanged(Object changedObj, Enum newState, Enum oldState) {
        if ( changedObj instanceof PduOutlet ) {            
            if ( newState == PduOutletState.OFF ) {
                PduOutlet outlet = (PduOutlet)changedObj;
                alertService.raiseAlert(new Alert(PduProperties.PDU_OUTLET_ALERT,"Outlet turned off"), AlertState.ALARM, "Outlet "+outlet.getName()+" has been turned off");
            }
        }
        if ( changedObj == this && newState != PduState.OK ) {
            alertService.raiseAlert(new Alert(PduProperties.PDU_ALERT,"PDU problem"), AlertState.ALARM, "PDU "+getName()+" is in state "+newState);            
        }
    }

   /**
    *  Checks a channel's parameters for validity.
    *
    *  @param  name     The channel name
    *  @param  hwChan   The hardware channel
    *  @param  type     The channel type string
    *  @param  subtype  The channel subtype string
    *  @return  Two-element array containing the encoded type [0] and subtype [1] 
    *  @throws  Exception  If parameters are invalid
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        Integer iType = typeMap.get(type.toUpperCase());
        int index = 0;
        if (iType == null) {
            MonitorLogUtils.reportError(sLog, name, "channel type", type);
        }
        if (iType == TYPE_OUTLET) {
            index = outlets.size();
            outlets.add(subtype);
        }
        else {
            if (hwChan != CHAN_CURRENT && hwChan != CHAN_POWER) {
                MonitorLogUtils.reportError(sLog, name, "hardware channel", hwChan);
            }
        }
        
        return new int[]{iType | (index << 16), 0};
    }


    /**
     *  Initializes a channel.
     *
     *  @param  name     The channel name
     *  @param  id       The channel ID
     *  @param  hwChan   The hardware channel
     *  @param  type     The encoded channel type
     *  @param  subtype  The encoded channel subtype
     */
    @Override
    protected void initChannel(String name, int id, int hwChan, int type, int subtype)
    {
        try {
            if ((type & 0xffff) == TYPE_OUTLET) {
                int index = type >> 16;
                String outlet = outlets.get(index);
                Integer outNum = outNumMap.get(outlet);
                if (outNum == null) {
                    MonitorLogUtils.reportError(sLog, name, "outlet name", outlet);
                }
                outNums[index] = outNum;
            }
        }
        catch (Exception e) {
            dropChannel(id);
        }
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan  The hardware channel number
     *  @param  type    The encoded channel type
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = 0;
        switch (type & 0xffff) {
        case TYPE_OUTLET:
            int outNum = outNums[type >> 16];
            try {
                value = pdu.isOutletOn(outNum) ? 1.0 : 0.0;
            }
            catch (DriverException e) {
                sLog.error("Error getting outlet state: " + e);
                setOnline(false);
            }
            break;

        case TYPE_POWER:
            try {
                value = hwChan == CHAN_POWER ? pdu.readPower() : pdu.readCurrent()[0];
            }
            catch (DriverException e) {
                sLog.error("Error reading power/current value: " + e);
                setOnline(false);
            }
        }
        return value;
    }

    @Override
    protected void readChannelGroup() {
        if ( readPduOutlets ) {
            for ( PduOutlet outlet: mapOfOutlets.values()) {
                try {
                    double value = pdu.isOutletOn(outlet.getIndex()) ? 1.0 : 0.0;
                    stateService.updateAgentComponentState(outlet, value == 1 ? PduOutletState.ON : PduOutletState.OFF);
                } catch (DriverException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    /**
     *  Disables the PDU.
     *
     *  Needed to relinquish control to the web or another telnet client
     */
    @Override
    @Command(name="disable", description="Disable the connection to the PDU")
    public void disable()
    {
        super.disable();
        setOnline(false);
    }


    /**
     *  Enables the PDU.
     */
    @Override
    @Command(name="enable", description="Enable the connection to the PDU")
    public void enable()
    {
        super.enable();
    }


    /**
     *  Gets the list of outlet names.
     *
     *  @return  The list of names
     */
    @Command(name="getOutletNames", description="Get the list of outlet names")
    public List<String> getOutletNames()
    {
        return new ArrayList<>(outNumMap.keySet());
    }


    /**
     *  Gets the map of outlet on states.
     *
     *  @return  The map of outlet names to on states (true or false)
     *  @throws  DriverException
     */
    @Command(name="getOutletOnStateMap", description="Get the map of outlet on states")
    public Map<String, Boolean> getOutletOnStateMap() throws DriverException
    {
        return pdu.getOutletOnStateMap();
    }


    /**
     *  Turns an outlet on.
     *
     *  @param  name  The outlet name
     *  @throws  DriverException
     */
    @Command(name="outletOn", description="Turn outlet on")
    public void outletOn(@Argument(name="name", description="Outlet name")
                         String name) throws DriverException
    {
        pdu.delayedOutletOn(getOutletNumber(name));        
    }


    /**
     *  Turns an outlet off.
     *
     *  @param  name  The outlet name
     *  @throws  DriverException
     */
    @Command(name="outletOff", description="Turn outlet off")
    public void outletOff(@Argument(name="name", description="Outlet name")
                          String name) throws DriverException
    {
        pdu.delayedOutletOff(getOutletNumber(name));
    }


    /**
     *  Forces an outlet to turn on.
     *
     *  @param  name  The outlet name
     *  @throws  DriverException
     */
    @Command(name="forceOutletOn", description="Force outlet to turn on")
    public void forceOutletOn(@Argument(name="name", description="Outlet name")
                              String name) throws DriverException
    {
        pdu.setOutletOn(getOutletNumber(name));
    }


    /**
     *  Forces an outlet to turn off.
     *
     *  @param  name  The outlet name
     *  @throws  DriverException
     */
    @Command(name="forceOutletOff", description="Force outlet to turn off")
    public void forceOutletOff(@Argument(name="name", description="Outlet name")
                               String name) throws DriverException
    {
        pdu.setOutletOff(getOutletNumber(name));
    }


    /**
     *  Gets whether an outlet is on.
     *
     *  @param  name  The outlet name
     *  @return  Whether outlet is turned on
     *  @throws  DriverException
     */
    @Command(name="isOutletOn", description="Get whether outlet is on")
    public boolean isOutletOn(@Argument(name="name", description="Outlet name")
                              String name) throws DriverException
    {
        return pdu.isOutletOn(getOutletNumber(name));
    }


    /**
     *  Change the state of an outlet
     *
     *  @param name  The outlet name
     *  @param state The desired PduOutletState for the outlet.
     *  @param force true if the state change is to be forced.
     *  @throws  DriverException
     */
    @Command(name="setOutletState", description="Set the outlet state to a given OutletState")
    public void setOutletState(
            @Argument(allowedValueProvider = "getOutletNames", name="name", description="Outlet name") String name, 
            @Argument(name="state", description="The desired PduOutletState") PduOutletState state, 
            @Argument(name="force", description="true to force the state change", defaultValue = "false") boolean force 
            ) throws DriverException
    {
        PduOutlet outlet = mapOfOutlets.get(name);
        if ( outlet == null ) {
            throw new IllegalArgumentException("Outlet name: "+name+" is illegal. The chosen outlet does not exist.");
        }
        if ( force ) {
            if ( state == PduOutletState.OFF ) {
                pdu.setOutletOff(name);
            } else {
                pdu.setOutletOn(name);                
            }
        } else {
            if ( state == PduOutletState.OFF ) {
                pdu.delayedOutletOff(name);
            } else {
                pdu.delayedOutletOn(name);                
            }            
        }
        stateService.updateAgentComponentState(outlet, state);
    }


    
    
    
    
    /**
     *  Gets an outlet number from its name.
     *
     *  This routine should be unnecessary since the PDU handles names directly,
     *  but there's a bug in the PDU parser which causes it to report an error
     *  if the name is hyphenated.
     *
     *  @param  name  The outlet name
     *  @return  The outlet number
     *  @throws  DriverException
     */
    int getOutletNumber(String name) throws DriverException
    {
        Integer outNum = outNumMap.get(name);
        if (outNum == null) {
            throw new DriverException("Invalid outlet name");
        }
        return outNum;
    }    
}
