package org.lsst.ccs.command;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import org.lsst.ccs.command.annotations.Command;

/**
 * A CompositeCommandSet that provides a route to activate another CommandSet.
 * When the RoutingCommandSet is active, it adds to its own command set
 * the route hidden CommandSet, making it available for straight execution.
 *
 * @author turri
 */
public class RoutingCommandSet implements CommandSet {

    private final RoutingDictionary dict;
    private final String route;
    private final String path;
    private final CommandSet cs;
    private final RouteSelectionCommandSet routeSelection;
    
    public RoutingCommandSet(String route, CommandSet cs) {
        this(null, route, route, cs);
    }

    public RoutingCommandSet(RouteSelectionCommandSet routeSelection, String route, CommandSet cs) {
        this(routeSelection, route, route, cs);
    }

    public RoutingCommandSet(RouteSelectionCommandSet routeSelection, String route, String path, CommandSet cs) {
        this.route = route;
        this.path = path;
        this.cs = cs;        
        dict = new RoutingDictionary(routeSelection, route, cs.getCommandDictionary());
        this.routeSelection = routeSelection;
    }

    String getRoute() {
        return route;
    }
    
    String getPath() {
        return path;
    }

    CommandSet getRouteCommandSet() {
        return cs;
    }

    @Override
    public Dictionary getCommandDictionary() {
        return dict;
    }

    @Override
    public Object invoke(BasicCommand command) throws CommandInvocationException, CommandArgumentMatchException, CommandArgumentTypeException {
        DictionaryCommand dc = dict.findCommand(command);
        if (dc == null) {
            throw new CommandInvocationException("Error: Could not invoke command " + command.getCommand() + " on target " + getRoute());
        }

        if (command.getArgumentCount() == 0) {
            throw new CommandInvocationException("Error: To change target type the command \"set target TARGET_NAME\"");
        }

        String route = command.getCommand();
        TokenizedCommand tc = (TokenizedCommand) command;
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < tc.getArgumentCount(); i++) {
            sb.append(tc.getArgument(i));
            sb.append(" ");
        }
        TokenizedCommand routedCommand = new TokenizedCommand(sb.toString());
        return getRouteCommandSet().invoke(routedCommand);

    }

    static class RoutingDictionary implements Dictionary {

        private static final long serialVersionUID = 1001943350979114050L;
        private final RoutingDictionaryHelpGenerator helpGenerator;
        private final Dictionary innerDictionary;
        private final RoutingDictionaryCompleter completer;
        private final RoutingCommand routingCommand;
        private final RouteSelectionCommandSet routeSelection;
        private final String route;

        RoutingDictionary(RouteSelectionCommandSet routeSelection, String route, Dictionary dict) {
            routingCommand = new RoutingCommand(route, dict);
            innerDictionary = dict;
            completer = new RoutingDictionaryCompleter(this);
            helpGenerator = new RoutingDictionaryHelpGenerator(routingCommand);
            this.routeSelection = routeSelection;
            this.route = route;
        }

        DictionaryCommand getRouteCommand(String route) {
            return routingCommand;
        }

        Dictionary getInnerDictionary() {
            return innerDictionary;
        }

        String getRoute() {
            return routingCommand.route;
        }

        @Override
        public boolean containsCommand(BasicCommand tc) {
            return routingCommand.route.equals(tc.getCommand());
        }

        @Override
        public DictionaryCommand findCommand(BasicCommand tc) {
            return containsCommand(tc) ? routingCommand : null;
        }

        @Override
        public int size() {
            return 1;
        }

        @Override
        public Iterator<DictionaryCommand> iterator() {
            Set<DictionaryCommand> set = new HashSet<>();
            set.add(routingCommand);
            return set.iterator();
        }

        @Override
        public DictionaryHelpGenerator getHelpGenerator() {
            return helpGenerator;
        }

        @Override
        public DictionaryCompleter getDictionaryCompleter() {
            return completer;
        }
        
        @Override
        public void setLevelForTypes(int level, Command.CommandType... types) {
           innerDictionary.setLevelForTypes(level,types);
        }

        @Override
        public int getLevelForType(Command.CommandType type) {
            return innerDictionary.getLevelForType(type);
        }

        @Override
        public void setCategoryVisible(Command.CommandCategory category, boolean isVisible) {
            innerDictionary.setCategoryVisible(category, isVisible);
        }
    
        @Override
        public boolean isCategoryVisible(Command.CommandCategory category) {
            return innerDictionary.isCategoryVisible(category);
        }

        @Override
        public boolean isDictionaryCommandVisible(DictionaryCommand dc) {
            if ( dc instanceof RoutingCommand ) {
                return ((RoutingCommand)dc).getArguments()[0].getAllowedValues().size() > 0;
            }
            return innerDictionary.isDictionaryCommandVisible(dc); //To change body of generated methods, choose Tools | Templates.
        }
        
        public boolean hasVisibleCommands() {
            Iterable<DictionaryCommand> iter = innerDictionary.filterByVisibilityIterator();
            while ( iter.iterator().hasNext() ) {
                return true;
            }
            return false;
        }
        
        public boolean isRouteAvailable() {
            String selectedRoute = routeSelection != null ? routeSelection.getActiveRoute() : "";
            if ( selectedRoute.isEmpty() ) {
                return !route.contains("/");
            } else {
                if ( route.startsWith(selectedRoute) ) {
                    String partialRoute = route.replace(selectedRoute, "");
                    if ( partialRoute.startsWith("/") ) {
                        partialRoute = partialRoute.substring(1);
                    }
                    return !partialRoute.contains("/");
                } else {
                    return false;
                }
                
            }
        }
    }

    private static class RoutingDictionaryCompleter implements DictionaryCompleter {

        private static final long serialVersionUID = 5979780403613106536L;
        private final DictionaryCompleter routingDictionaryCompleter;
        private final DictionaryCompleter innerDictionaryCompleter;
        private final String route;
        
        RoutingDictionaryCompleter(RoutingDictionary dict) {
            routingDictionaryCompleter = new DefaultDictionaryCompleter(dict);
            innerDictionaryCompleter = dict.getInnerDictionary().getDictionaryCompleter();
            route = dict.getRoute();
        }

        @Override
        public int complete(String buffer, int index, List<CharSequence> list) {
            TokenizedCommand tc = new TokenizedCommand(buffer.substring(0, index));
            int argCount = tc.getArgumentCount();
            boolean endsWithWhiteSpace = !tc.isEmpty() && Character.isWhitespace(buffer.charAt(index - 1));

            if (endsWithWhiteSpace) {
                argCount++;
            }
            if (argCount > 1 && buffer.startsWith(route)) {
                if (innerDictionaryCompleter != null) {
                    int extraIndex = route.length() + 1;
                    String newBuffer = buffer.substring(extraIndex);
                    int newIndex = index - extraIndex;

                    int completionValue = innerDictionaryCompleter.complete(newBuffer, newIndex, list);
                    return completionValue == -1 ? completionValue : completionValue + route.length() + 1;
                }
            } else {
                int ret = routingDictionaryCompleter.complete(buffer, index, list);
                return ret;
            }
            return index;
        }

    }

    private static class RoutingDictionaryHelpGenerator implements DictionaryHelpGenerator {

        private static final long serialVersionUID = 2986754545846150826L;
        private final RoutingCommand routingCommand;

        RoutingDictionaryHelpGenerator(RoutingCommand routingCommand) {
            this.routingCommand = routingCommand;
        }

        @Override
        public boolean hasHelp(DictionaryCommand command) {
            return routingCommand.equals(command);
        }

        @Override
        public String modifyHelpForCommand(DictionaryCommand command, String help, boolean compact) {
            help = "(t) " + help;
            if (!compact) {
                help += "Available commands: \n";
                help += DictionaryUtils.basicHelpForDictionary(((RoutingCommand) command).getDictionary(), "     ");
            }
            return help;
        }
    }

    private static class RoutingCommand implements DictionaryCommand {

        private static final long serialVersionUID = 5566879570663731785L;
        private final String route;
        private final Dictionary dict;
        private final DictionaryArgument[] routingArgument;
        private final int level;

        RoutingCommand(String route, Dictionary dict) {
            this.route = route;
            this.dict = dict;
            routingArgument = new DictionaryArgument[]{new RoutingCommandArgument(dict), new RoutingCommandArgumentsArgument()};
            int l = Command.NOT_DEFINED;
        
            // the routing command level is the minimum required level for this 
            // route to have at least one command
            for (DictionaryCommand dc : dict) {
                if (dc.getLevel() < l) {
                    l = dc.getLevel();
                }
            }
            level = l;
        }

        Dictionary getDictionary() {
            return dict;
        }

        @Override
        public DictionaryArgument[] getArguments() {
            return routingArgument;
        }

        @Override
        public String getCommandName() {
            return route;
        }

        @Override
        public String getDescription() {
            return "Sends commands to target's (" + route + ") dictionary";
        }

        @Override
        public Command.CommandType getType() {
            return Command.CommandType.QUERY;
        }

        @Override
        public boolean isVarArgs() {
            return true;
        }

        @Override
        public int getLevel() {
            return level;
        }

        @Override
        public Command.CommandCategory getCategory() {
            return Command.CommandCategory.USER;
        }

        @Override
        public boolean isAutoAck() {
            return true;
        }

        @Override
        public Duration getTimeout() {
            return null;
        }
        
        private static class RoutingCommandArgumentsArgument implements DictionaryArgument {

            private static final long serialVersionUID = 6523660726535011547L;
            
            @Override
            public String getDescription() {
                return "The arguments for the command";
            }

            @Override
            public String getName() {
                return "arguments...";
            }

            @Override
            public String getSimpleType() {
                return "String";
            }

            @Override
            public String getType() {
                return "java.lang.String";
            }

            @Override
            public List<String> getAllowedValues() {
                return Collections.EMPTY_LIST;
            }

            @Override
            public String getDefaultValue() {
                return null;
            }

        }

        private static class RoutingCommandArgument implements DictionaryArgument {

            private static final long serialVersionUID = -7280662026390377893L;
            private final Dictionary dict;

            RoutingCommandArgument(Dictionary dict) {
                this.dict = dict;
            }

            @Override
            public String getDescription() {
                return "The command to execute";
            }

            @Override
            public String getName() {
                return "command";
            }

            @Override
            public String getSimpleType() {
                return "String";
            }

            @Override
            public String getType() {
                return "java.lang.String";
            }

            @Override
            public List<String> getAllowedValues() {
                ArrayList<String> allowedValues = new ArrayList<>();
                for (DictionaryCommand command : dict.filterByVisibilityIterator()) {
                        allowedValues.add(command.getCommandName());
                }
                return allowedValues;
            }

            @Override
            public String getDefaultValue() {
                return null;
            }

        }
    }
}
