package org.lsst.ccs.command;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.Collections;
import java.util.List;
import org.lsst.ccs.utilities.conv.InputConversionEngine;

/**
 * Takes a single object and builds a command set from its annotated methods.
 * The command set consists of two parts, the CommandDictionary, which is
 * serializable and the command invoker, which contains references to the object
 * and methods and is not serializable.
 *
 * @author tonyj
 */
public class CommandSetBuilder {

    private final InputConversionEngine engine = new InputConversionEngine();

    /**
     * Build a command set from an objects annotations.
     *
     * @param object The object from which annotations will be extracted, and to
     * which invocation requests will be forwarded.
     * @return A CommandSet representing the commands found in the object.
     */
    public CommandSet buildCommandSet(Object object) {
        Class targetClass = object.getClass();
        CommandDictionaryBuilder dictionary = new CommandDictionaryBuilder(targetClass);
        CommandSetImplementation commandSet = new CommandSetImplementation(dictionary, object);

        // Adding allowedValueProvider to the list of DictionaryCommand. We need the object itself.
        for (DictionaryCommand dc : dictionary.getCommandDictionary()) {
            for (DictionaryArgument da : dc.getArguments()) {
                if (da instanceof MethodBasedDictionaryArgument) {
                    MethodBasedDictionaryArgument mbda = (MethodBasedDictionaryArgument) da;
                    if (mbda.getAllowedValueMethod() != null) {
                        (mbda).setAllowedValuesProvder(
                                () -> {
                                try {
                                    return (List<String>) mbda.getAllowedValueMethod().invoke(object);
                                } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException ex) {
                                    return Collections.emptyList();
                                }
                            }
                        );
                    }
                }
            }
        }
        return commandSet;
    }

    private class CommandSetImplementation implements CommandSet {

        private final CommandDictionaryBuilder dict;
        private final Object target;

        private CommandSetImplementation(CommandDictionaryBuilder dict, Object target) {
            this.dict = dict;
            this.target = target;
        }

        @Override
        public Dictionary getCommandDictionary() {
            return dict.getCommandDictionary();
        }

        @Override
        public Object invoke(BasicCommand command) throws CommandInvocationException, CommandArgumentMatchException, CommandArgumentTypeException {
            Method method = dict.getMethod(command);
            if (method == null) {
                throw new CommandInvocationException("Error: No handler found for command %s with %d arguments", command.getCommand(), command.getArgumentCount());
            }
            return invoke(target, method, command);
        }

        private Object invoke(Object target, Method method, BasicCommand command) throws CommandInvocationException, CommandArgumentTypeException {

            RawCommand raw = RawCommand.toRawCommand(command, method, engine);
            try {
                return method.invoke(target, raw.getArguments());
            } catch (IllegalAccessException | IllegalArgumentException ex) {
                String msg = "Error: Can't invoke command " + command.prettyToString() + "\n";
                msg += "Attempted to invoke it on method " + method.getName() + " on object " + target.getClass().getCanonicalName() + " (" + target + ")\n";
                int pars = method.getParameterCount();
                msg += "Method " + method.getName() + " requires " + pars + " parameters of type: \n";
                for (int i = 0; i < pars; i++) {
                    Parameter p = method.getParameters()[i];
                    msg += "-" + i + "- " + p.getName() + " " + p.getType().getCanonicalName() + "\n";
                }
                msg += "Command " + command.getCommand() + " has " + command.getArgumentCount() + " arguments\n";
                for (int i = 0; i < command.getArgumentCount(); i++) {
                    Object a = command.getArguments()[i];
                    msg += "-" + i + "- " + a + " " + a.getClass().getCanonicalName() + "\n";
                }
                throw new CommandInvocationException(msg, ex);
            } catch (InvocationTargetException ex) {
                throw new CommandInvocationException(ex);
            }
        }
    }
}
