package org.lsst.ccs.command;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Set;
import org.lsst.ccs.command.annotations.Option;

/**
 * A Class containing the set of user specified options.
 * 
 * An instance of this class is passed during method invocation of a command
 * that supports options.
 * 
 */
public class Options implements Serializable {

    private final Set<String> options = new HashSet<>();
    private static final long serialVersionUID = 345738952754085472L;

    //Default constructor
    public Options() {        
    }
    
    //Copy constructor
    public Options(Options o) {
        this.options.addAll(o.options);
    }

    public Options withOption(String opt) {
        options.add(opt);
        return this;
    }
    
    /**
     * Check the presence of an option.
     * @param option
     * @return
     * @deprecated use hasOption(SupportedOption) instead
     */
    public boolean hasOption(String option) {
        return hasOption(SupportedOption.getSupportedOption(option));
    }

    public boolean hasOption(SupportedOption option) {
        return options.contains(option.getSingleLetterName()) || options.contains(option.getName());
    }
    
    @Override
    public String toString() {
        return "Options{" + "options=" + options + '}';
    }
       
    public boolean hasOptions() {
        return !options.isEmpty();
    }
    
    public Set<String> getOptions() {
        return new LinkedHashSet<>(options);
    }
    
    /** 
     * Remove an option. This method works in conjunction with method
     * DictionaryCommand::addSupportedOption. When options are added
     * externally, they will also have to be removed once they are
     * processed by the entity that added the option in the first place.
     * 
     * @param option 
     */
    public void removeOption(SupportedOption option) {
        options.remove(option.getName());
        options.remove(option.getSingleLetterName());
    }
    
}
