package org.lsst.ccs.command.annotations;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.*;

/**
 * Annotation for commands. Allows to specify the name of a command, otherwise
 * method's name is used.
 *
 * @author turri
 */
@Target(ElementType.METHOD)
@Retention(RetentionPolicy.RUNTIME)
@Documented
public @interface Command {

    public static enum CommandType {
        QUERY,
        ACTION,
        CONFIGURATION,
        SIGNAL,
        @Deprecated //Replace with SIGNAL
        ABORT
    }
    
    public static enum CommandCategory {
        /** 
         * System commands are not normally directly invoked by users, and
         * so are normally hidden in help. For example commands used by GUIs 
         * fall into this category.
         */
        SYSTEM,
        /**
         * CORE commands are defined by the core.
         */
        CORE,
        /**
         * USER commands are defined by user-level code.
         * This is the default.
         */
        USER
    }

    // Some pre-defined levels, this list is not necessarily exhaustive, which is
    // why we do not use an enumeration.
    public static final int NORMAL = 0;
    @Deprecated
    public static final int ENGINEERING1 = 1;
    public static final int ENGINEERING_ROUTINE = 1;
    @Deprecated
    public static final int ENGINEERING2 = 2;
    public static final int ENGINEERING_ADVANCED = 2;
    @Deprecated
    public static final int ENGINEERING3 = 3;
    public static final int ENGINEERING_EXPERT = 3;
    /** Users whose max allowed level is at least ADMIN can perform lock management administrative actions, like destroying other user's locks. */
    public static final int ADMIN = 50;
    /** Maximum valid level. */
    public static final int MAX = 99; 
    public static final int NOT_DEFINED = 99999;

    /**
     * If not null it will replace the method's name as the command name.
     *
     * @return "" or null if default name is used, user-specified name
     * otherwise.
     */
    String name() default ""; // if "" then Null is assumed.

    /**
     * Specify the description of the command.
     *
     * @return command's description or "" if not set.
     */
    String description() default "";

    /**
     * Specify aliases for the command.
     * Multiple aliases can be specified as a comma separated list.
     *
     * @return command's abbreviation(s) or "" if not set.
     */
    String alias() default "";

    /**
     * The CommandType of the Command
     *
     * @return The command type.
     */
    CommandType type() default CommandType.ACTION;

    /**
     * Specify the level of the command. Only locks at the given level, or
     * higher will have access to the command. Valid level range: 0 - 99.
     *
     * @return the level of the Command. The default is NOT_DEFINED.
     */
    int level() default NOT_DEFINED;
    
    /**
     * 
     * @return Whether this is a system command or not
     */
    CommandCategory category() default CommandCategory.USER;
    
    /**
     * Returns {@code true} if the ACK should be automatically sent by the framework 
     * before invoking this command.
     * 
     * @return {@code true} if the ACK should be automatically generated.
     */
    boolean autoAck() default true;
    
    /**
     * Timeout for the command in seconds.
     * Non-positive value indicates that the default timeout of the command originator should be used.
     * 
     * @return Timeout value in seconds.
     */
    int timeout() default 0;
    
    /**
     * Is this a command for simulation only?
     * Simulation only commands will be added to the object's dictionary
     * only when property org.lsst.ccs.run.mode is set to "true". 
     * 
     * @return Timeout value in seconds.
     */
    boolean simulation() default false;
    
    /** Handles string representations of command levels. */
    static public class Level {
        
        static private final SortedMap<Integer,String> value2name;
        static private final Map<String,Integer> name2value;
        
        static public enum OnTooHigh {ON_TOO_HIGH_EXCEPTION, ON_TOO_HIGH_DEFAULT, ON_TOO_HIGH_MAX}
        static public enum OnInvalidName {ON_INVALID_NAME_EXCEPTION, ON_INVALID_NAME_DEFAULT}
        static public enum OnUnnamed {ON_UNNAMED_EXCEPTION, ON_UNNAMED_DEFAULT, ON_UNNAMED_NUMERIC}
        
        private int defaultValue = -1;
        private String defaultName = null;
        private OnTooHigh onTooHigh = OnTooHigh.ON_TOO_HIGH_MAX;
        private OnInvalidName onInvalidName = OnInvalidName.ON_INVALID_NAME_DEFAULT;
        private OnUnnamed onUnnamed = OnUnnamed.ON_UNNAMED_NUMERIC;
        
        public int getValue(String name) {
            Integer out = name2value.get(name);
            if (out == null) {
                try {
                    int value = Integer.parseInt(name);
                    if (value < 0) throw new NumberFormatException();
                    if (value > MAX) {
                        return switch(onTooHigh) {
                            case ON_TOO_HIGH_DEFAULT -> defaultValue;
                            case ON_TOO_HIGH_MAX -> MAX;
                            default -> throw new IllegalArgumentException("Illegal level value "+ name);
                        };
                    } else {
                        return value;
                    }
                } catch (NumberFormatException x) {
                    return switch(onInvalidName) {
                        case ON_INVALID_NAME_DEFAULT -> defaultValue;
                        default -> throw new IllegalArgumentException("Invalid level name "+ name);
                    };
                }
            }
            return out;
        }
        
        public String getName(int value) {
            String out = valueToName().get(value);
            if (out == null) {
                return switch (onUnnamed) {
                    case ON_UNNAMED_EXCEPTION -> throw new IllegalArgumentException("Unnamed level "+ value);
                    case ON_UNNAMED_DEFAULT -> defaultName;
                    default -> {
                        if (value > MAX) {
                            yield switch(onTooHigh) {
                                case ON_TOO_HIGH_DEFAULT -> defaultName;
                                case ON_TOO_HIGH_MAX -> "MAX";
                                default -> throw new IllegalArgumentException("Illegal level value " + value);
                            };
                        } else {
                            yield String.valueOf(value);
                        }
                    }
                };
            }
            return out;
        }
        
        public Level with(Enum<?> what) {
            if (what instanceof OnTooHigh w) {
                onTooHigh = w;
            } else if (what instanceof OnInvalidName w) {
                onInvalidName = w;
            } else if (what instanceof OnUnnamed w) {
                onUnnamed = w;
            } else {
                throw new IllegalArgumentException("Illegal argument type: "+ (what == null ? null : what.getClass().getName()));
            }
            return this;
        }
        
        public Level withDefault(int defaultValue) {
            this.defaultValue = defaultValue;
            return this;
        }
        
        public Level withDefault(String defaultName) {
            this.defaultName = defaultName;
            return this;
        }
        
        static {
            TreeMap<Integer,String> v2n = new TreeMap<>();
            for (Field f : Command.class.getDeclaredFields()) {
                try {
                    if (!f.isAnnotationPresent(Deprecated.class)) {
                        int mm = f.getModifiers();
                        if (f.getType().equals(int.class) && Modifier.isFinal(mm) && Modifier.isStatic(mm) && Modifier.isPublic(mm)) {
                            v2n.put(f.getInt(null), f.getName());
                        }
                    }
                } catch (RuntimeException | ReflectiveOperationException xx) {
                }
            }
            value2name = Collections.unmodifiableSortedMap(v2n);
            
            LinkedHashMap<String,Integer> n2v = new LinkedHashMap<>(1+Math.round(v2n.size()/.75f));
            v2n.forEach((value,name) -> {
                n2v.put(name, value);
            });
            name2value = Collections.unmodifiableMap(n2v);
        }
    
        /**
         * Parses string representation of a command level.
         * 
         * @param s String representation.
         * @param def Value to be returned if the provided string is not parsable.
         * @return Numeric command level.
         */
        static public int parse(String s, int def) {
            int out = -1;
            if (s != null) {
                s = s.trim();
                try {
                    out = Integer.parseInt(s);
                } catch (NumberFormatException x) {
                    try {
                        Field f = Command.class.getDeclaredField(s);
                        int mm = f.getModifiers();
                        if (f.getType().equals(int.class) && Modifier.isFinal(mm) && Modifier.isStatic(mm) && Modifier.isPublic(mm)) {
                            out = f.getInt(null);
                        }
                    } catch (RuntimeException | ReflectiveOperationException xx) {
                    }
                }
            }
            if (out < 0) {
                out = def;
            } else if (out > MAX && out != NOT_DEFINED) {
                out = MAX;
            }
            return out;
        }
        
        static public String name(int level, boolean numberAsString) {
            String out = valueToName().get(level);
            if (out == null) {
                if (level == NOT_DEFINED) {
                    out = "NOT_DEFINED";
                } else if (numberAsString) {
                    if (level < 0) {
                        out = "NOT_DEFINED";
                    } else if (level > MAX) {
                        out = "MAX";
                    } else {
                        out = Integer.toString(level);
                    }
                }
            }
            return out;
        }
        
        static public Map<String,Integer> nameToValue() {
            return name2value;
        }
        
        static public SortedMap<Integer,String> valueToName() {
            return value2name;
        }
        
    }
    
}
