package org.lsst.ccs.command;

import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import org.lsst.ccs.command.StringTokenizer.Token;

/**
 * A command line that has been split into tokens. This could in future be
 * expanded to also support command options (beginning with - or --)
 *
 * @author tonyj
 */
public class TokenizedCommand implements BasicCommand {

    private final List<Token> tokens;
    private final String[] args;
    private static final long serialVersionUID = -687119049448004978L;
    private final Options options = new Options();

    private final List<Token> argumentTokenList = new ArrayList<>();
    
    private Token lastToken;

    /**
     * Builds a Tokenized command from an unparsed string.
     *
     * @param command The command string to be tokenized.
     */
    public TokenizedCommand(String command) {
        tokens = StringTokenizer.tokenize(command);
        
        List<String> argumentValues = new ArrayList<>();
        
        boolean isFirst = true;        
        for (Token token : tokens) {
            if ( isFirst ) {
                //This is the command
                isFirst = false;
            } else {                
                if (token.isOption()) {
                    if ( token.isShortOption() ) {
                        String shortOptions = token.getString();
                        for ( int ii = 0; ii< shortOptions.length(); ii++ ) {
                            options.withOption(Character.toString(shortOptions.charAt(ii)));
                        }                        
                    } else {
                        options.withOption(token.getString());
                    }
                } else {
                    argumentTokenList.add(token);
                    argumentValues.add(token.getString());
                }
            }
            lastToken = token;
        }
        
        args = new String[argumentValues.size()];
        argumentValues.toArray(args);
        
    }

    /**
     * Get the root command name (the zeroth token)
     *
     * @return The command name
     */
    @Override
    public String getCommand() {
        return tokens.get(0).getString();
    }

    /**
     * Get the start position on the line of the command name.
     *
     * @return The position of the command
     */
    int getCommandLocation() {
        return tokens.get(0).getLocation();
    }

    /**
     * Get a specific command argument
     *
     * @param index The index of the argument
     * @return The command argument at the given index
     */
    @Override
    public String getArgument(int index) {
        return args[index];
    }

    @Override
    public String[] getArguments() {
        return args;
    }
    
    /**
     * The start point in the original command line of a specific argument.
     *
     * @param index The index of the argument
     * @return The position of the argument
     */
    int getArgumentLocation(int index) {
        return argumentTokenList.get(index).getLocation();
    }

    /**
     * Get the number of arguments associated with this command
     *
     * @return The argument count
     */
    @Override
    public int getArgumentCount() {
        return args.length;
    }

    /**
     * Test if the command is completely empty (no tokens)
     *
     * @return <code>true</code> only if the command is empty.
     */
    public boolean isEmpty() {
        return tokens.isEmpty();
    }
    
    @Override
    public String toString() {
        return prettyToString();
    }

    @Override
    public Options getOptions() {
        //This is needed to support deserialization from older
        //versions that still don't support Options.
        if ( options == null ) {
            return new Options();
        }
        return options;
    }
    
    Token getLastToken() {
        return lastToken;
    }

}
