package org.lsst.ccs.command;

import java.io.Serializable;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.lsst.ccs.command.annotations.Command;

/**
 * Encapsulate the dictionary information for a single command and parameters.
 * This class is serializable for use in client-server applications, so does not
 * maintain any references to Class or Method objects which may not be available
 * in a remote client.
 * @author tonyj
 */
public interface DictionaryCommand extends Serializable {

    public final static String[] NO_ALIASES = new String[0];

    /**
     * Get the name of the DictionaryCommand. A DictionaryCommand can also have aliases.
     * See {@link #getAliases getAliases}.
     * 
     * @return The name of the Command.
     */
    public String getCommandName();
    
    /**
     * Get the array of aliases for this command. The DictionaryCommand can be referenced
     * either via its name or one of its aliases (if present).
     * 
     * @return The array of aliases. By default it returns an empty String array.
     */
    public default String[] getAliases() {
        return NO_ALIASES;
    }

    /**
     * Get the array of {@link DictionaryArgument DictionaryArgument} for this DictionaryCommand.
     * 
     * @return The array of DictionaryArguments.
     */
    public DictionaryArgument[] getArguments();

    /**
     * Get the description for this DictionaryCommand.
     * 
     * @return The DictionaryCommand description. By default it's the empty String.
     */
    public default String getDescription() {
        return "";
    }

    /**
     * Get the {@link Command.CommandType CommandType} for this DictionaryCommand.
     * 
     * @return The CommandType
     */
    public Command.CommandType getType();
    
    /**
     * Determine if this DictionaryCommand has a variable number of arguments.
     * 
     * @return <b>true</b> if this DictionaryCommand has a variable number of
     *         arguments. <b>false</b> otherwise.
     */
    public boolean isVarArgs();

    /**
     * Get the level for this DictionaryCommand. The higher the level the more
     * permissions are required to be able to invoke the command.
     * 
     * @return The level of the DictionaryCommand.
     */
    public int getLevel();
    
    /**
     * Test if value matches this command (either the command itself or one if
     * its aliases).
     *
     * @param value The value to test
     * @return <code>true</code> if the value matches
     */
    public default boolean matchesCommandOrAlias(String value) {
        if (getCommandName().equals(value)) {
            return true;
        }
        for (String alias : getAliases()) {
            if (alias.equals(value)) {
                return true;
            }
        }
        return false;        
    }
    
    /**
     * Get the {@link Command.CommandCategory CommandCategory} for this DictionaryCommand.
     * The {@link Command.CommandCategory CommandCategory} is used to determine the visibility
     * of this command during tab completion or when the help command is invoked.
     * By default DictionaryCommands with category USER are displayed.
     * 
     * @return The DictionaryCommand {@link Command.CommandCategory CommandCategory}
     */
    public Command.CommandCategory getCategory();
    
    /**
     * Returns {@code true} if the ACK should be automatically sent by the framework 
     * before invoking this command.
     * 
     * @return {@code true} if the ACK should be automatically generated.
     */
    public boolean isAutoAck();
    
    /**
     * Returns the default timeout value associated with this command.
     * @return The default timeout for this command, or {@code null} if there is no default.
     */
    public Duration getTimeout();
    
    
    /**
     * Does this Command have options?
     * 
     * @return true if this Command supports Options
     */
    public default boolean hasOptions() {
        return !getSupportedOptions().isEmpty();
    }
        
    /**
     * Get the SupportedOptions for this command.
     * 
     * If method hasOptions() returns true the the SupportedOptions returned
     * by this method will contain the supported options, otherwise it
     * will be empty.
     * 
     * @return The SupportedOptions object for this command. If no options are
     *         supported, the returned List of SupportedOptions will be empty.
     */
    public default List<SupportedOption> getSupportedOptions() {
        return new ArrayList<>();
    }
    
    /**
     * Add an external Option not defined in the annotations.
     * This method works in conjunction with Options::removeOption.
     * After the external Option has been processed, it will have to be removed
     * or it will trigger exceptions as it will not be recognized by the
     * original @Command annotated method.
     *
     * @param opt The SupportedOption to be added.
     */
    public default void addSupportedOption(SupportedOption opt) {
        getSupportedOptions().add(opt);
    }

}
