package org.lsst.ccs.command;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArraySet;
import org.lsst.ccs.command.annotations.Command;

/**
 * A command dictionary to which other command dictionaries can be added and
 * removed.
 *
 * @author tonyj
 */
public class CompositeCommandDictionary implements Dictionary {

    private static final long serialVersionUID = 6746559441105266344L;
    private final Set<Dictionary> dicts = new CopyOnWriteArraySet<>();
    private final CompositeCommandDictionaryCompleter completer = new CompositeCommandDictionaryCompleter(dicts);
    private transient Map<String, Boolean> visibilityByCategory = new ConcurrentHashMap<>();

    public void add(Dictionary commandDictionary) {
        synchronized(dicts) {
            dicts.add(commandDictionary);
            for (String cat : visibilityByCategory.keySet()) {
                commandDictionary.setCategoryVisible(Command.CommandCategory.valueOf(cat), visibilityByCategory.get(cat));
            }
        }
    }

    public void remove(Dictionary commandDictionary) {
        dicts.remove(commandDictionary);
    }

    void clear() {
        dicts.clear();
    }

    @Override
    public DictionaryCompleter getDictionaryCompleter() {
        return completer;
    }

    @Override
    public boolean containsCommand(BasicCommand tc) throws CommandArgumentMatchException {
        List<CommandArgumentMatchException> exceptions = new ArrayList<>();
        for (Dictionary dict : dicts) {
            try {
                if (dict.containsCommand(tc)) {
                    return true;
                }
            } catch (CommandArgumentMatchException ex) {
                exceptions.add(ex);
            }
        }
        CommandArgumentMatchException.throwExceptionIfNeeded(exceptions);
        return false;
    }

    @Override
    public DictionaryCommand findCommand(BasicCommand tc) throws CommandArgumentMatchException {
        List<DictionaryCommand> matches = new ArrayList<>();
        List<CommandArgumentMatchException> exceptions = new ArrayList<>();
        for (Dictionary dict : dicts) {
            try {
                if (dict.containsCommand(tc)) {
                    matches.add(dict.findCommand(tc));
                }
            } catch (CommandArgumentMatchException ex) {
                exceptions.add(ex);
            }
        }
        if (matches.isEmpty()) {
            CommandArgumentMatchException.throwExceptionIfNeeded(exceptions);
            return null;
        } else if (matches.size() == 1) {
            return matches.get(0);
        } else {
            throw new AmbiguousCommandException("Found " + matches.size() + " matches when looking for command " + tc.getCommand() + " with " + tc.getArgumentCount() + " arguments");
        }
    }

    @Override
    public Iterator<DictionaryCommand> iterator() {
        // Brute force implementation, could do better
        ArrayList<DictionaryCommand> allCommands = new ArrayList<>();
        for (Dictionary dict : dicts) {
            for (DictionaryCommand def : dict) {
                allCommands.add(def);
            }
        }
        return allCommands.iterator();
    }

    @Override
    public int size() {
        int result = 0;
        for (Dictionary dict : dicts) {
            result += dict.size();
        }
        return result;
    }

    @Override
    public DictionaryHelpGenerator getHelpGenerator() {
        return new CompositeCommandDictionaryHelpGenerator(dicts);
    }


    public Set<Dictionary> getDictionaries() {
        return dicts;
    }

    @Override
    public void setLevelForTypes(final int level, Command.CommandType... types) {
        final Command.CommandType[] innerTypes = types == null || types.length == 0 ? Command.CommandType.values() : types;
        dicts.forEach((d) -> d.setLevelForTypes(level,innerTypes));
    }

    //What should this return? The first defined level?
    //The minimum level? Tha maximum?
    @Override    
    public int getLevelForType(Command.CommandType type) {                
        for (Dictionary dict : dicts) {   
            int level = dict.getLevelForType(type);
            if ( level != Command.NOT_DEFINED ) {
                return level;
            } 
        }
        return Command.NOT_DEFINED;
    }

    @Override
    public void setCategoryVisible(Command.CommandCategory category, boolean isVisible) {
        synchronized(dicts) {
            visibilityByCategory.put(category.name(), isVisible);
            dicts.forEach((d) -> d.setCategoryVisible(category,isVisible));            
        }
    }

    @Override
    public boolean isCategoryVisible(Command.CommandCategory category) {
        synchronized(dicts) {
            if (visibilityByCategory.containsKey(category.name()) ) {
                return visibilityByCategory.get(category.name());
            }
            for (Dictionary dict : dicts) {
                boolean isVisible = dict.isCategoryVisible(category);
                if (!isVisible) {
                    return false;
                }
            }
            return true;
        }
    }


    
    private static class CompositeCommandDictionaryHelpGenerator implements DictionaryHelpGenerator {

        private static final long serialVersionUID = -960170507549440673L;
        private final Set<Dictionary> dictionaries;

        CompositeCommandDictionaryHelpGenerator(Set<Dictionary> dicts) {
            this.dictionaries = dicts;
        }

        @Override
        public boolean hasHelp(DictionaryCommand command) {
            for (Dictionary dic : dictionaries) {
                DictionaryHelpGenerator helpGenerator = dic.getHelpGenerator();
                if (helpGenerator != null && helpGenerator.hasHelp(command)) {
                    return true;
                }
            }
            return false;
        }

        @Override
        public String modifyHelpForCommand(DictionaryCommand command, String help, boolean compact) {
            //Can multiple DictionaryHelpGenerators contribute to a DictionaryCommand help?
            for (Dictionary dic : dictionaries) {
                DictionaryHelpGenerator helpGenerator = dic.getHelpGenerator();
                if (helpGenerator != null && helpGenerator.hasHelp(command)) {
                    help = helpGenerator.modifyHelpForCommand(command, help, compact);
                    //If we want only one contribution, we have to break here.
                    //break;
                }
            }
            return help;
        }

    }

    private static class CompositeCommandDictionaryCompleter implements DictionaryCompleter {

        private static final long serialVersionUID = -6939187939493084396L;
        private final Set<Dictionary> dicts;

        CompositeCommandDictionaryCompleter(Set<Dictionary> dicts) {
            this.dicts = dicts;
        }

        @Override
        public int complete(String buffer, int cursor, List<CharSequence> list) {
            List<CharSequence> compositeCompletionList = new ArrayList<>();
            int returnValue = -1;
            for (Dictionary dict : dicts) {
                List<CharSequence> innerCompletionList = new ArrayList<>();
                DictionaryCompleter innerCompleter = dict.getDictionaryCompleter();
                if (innerCompleter != null) {
                    int completionValue = innerCompleter.complete(buffer, cursor, innerCompletionList);
                    if (completionValue >= returnValue && completionValue != -1) {
                        if (completionValue > returnValue) {
                            compositeCompletionList.clear();
                        }
                        compositeCompletionList.addAll(innerCompletionList);
                        returnValue = completionValue;
                    }
                }
            }
            list.addAll(compositeCompletionList);
            return returnValue;
        }

    }
    
    private void readObject(java.io.ObjectInputStream stream)
            throws IOException, ClassNotFoundException {
        stream.defaultReadObject();
        if ( visibilityByCategory == null ) {
            visibilityByCategory = new ConcurrentHashMap<>();
        }
    }
    

}
