package org.lsst.ccs.command;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.lsst.ccs.command.annotations.Command;

/**
 * An implementation of Dictionary which implements commands by linking to
 * annotated methods on an underlying command.
 *
 * @author tonyj
 */
class MethodBasedCommandDictionary extends ArrayList<DictionaryCommand> implements Dictionary {

    private static final long serialVersionUID = 4401762766383746012L;
    private final DefaultDictionaryCompleter completer = new DefaultDictionaryCompleter(this);
    
    private Map<Command.CommandType, Integer> levelsByType = new ConcurrentHashMap();
    private Map<String, Boolean> visibilityByCategory = new ConcurrentHashMap();
    
    MethodBasedCommandDictionary() {
        setDefaultVisibilityForCategories();
    }
    
    private void setDefaultVisibilityForCategories() {
        visibilityByCategory.put(Command.CommandCategory.USER.name(), Boolean.TRUE);
        visibilityByCategory.put(Command.CommandCategory.CORE.name(), Boolean.TRUE);
        visibilityByCategory.put(Command.CommandCategory.SYSTEM.name(), Boolean.FALSE);        
    }
    
    @Override
    public boolean containsCommand(BasicCommand tc) throws CommandArgumentMatchException {
        return findCommand(tc) != null;
    }

    @Override
    public DictionaryCommand findCommand(BasicCommand tc) throws CommandArgumentMatchException {
        String command = tc.getCommand();
        int argumentCount = tc.getArgumentCount();
        ArrayList<DictionaryCommand> matches = new ArrayList<>();

        ArrayList<CommandArgumentMatchException> exceptions = new ArrayList<>();

        for (DictionaryCommand def : this) {
            try {
                if (DictionaryUtils.commandMatch(def, tc)) {
                    matches.add(def);
                }
            } catch (CommandArgumentMatchException ex) {
                exceptions.add(ex);
            }
        }
        if (matches.isEmpty()) {
            CommandArgumentMatchException.throwExceptionIfNeeded(exceptions);
            return null;
        } else if (matches.size() == 1) {
            return matches.get(0);
        } else {
            throw new AmbiguousCommandException("Error finding command " + command + " with " + argumentCount + " arguments: " + matches.size() + " matches found in MethodBasedCommandDictionary");
        }
    }

    @Override
    public DictionaryCompleter getDictionaryCompleter() {
        return completer;
    }

    @Override
    public void setLevelForTypes(int level, Command.CommandType... types) {
        if ( types == null || types.length == 0) {
            types = Command.CommandType.values();
        }
        for(Command.CommandType type : types) {
            levelsByType.put(type, level);
        }
    }

    @Override
    public int getLevelForType(Command.CommandType type) {
        return levelsByType.getOrDefault(type,Command.NOT_DEFINED);
    }

    @Override
    public boolean isCategoryVisible(Command.CommandCategory category) {
        return visibilityByCategory.get(category.name());
    }

    @Override
    public void setCategoryVisible(Command.CommandCategory category, boolean isVisible) {
        visibilityByCategory.put(category.name(), isVisible);
    }


    private void writeObject(java.io.ObjectOutputStream out) throws IOException {
        levelsByType.clear();
        visibilityByCategory.clear();
        out.defaultWriteObject();
    }

    private void readObject(java.io.ObjectInputStream in) 
     throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        if (levelsByType == null) {
            levelsByType = new ConcurrentHashMap();
        }
        if (visibilityByCategory == null) {
            visibilityByCategory = new ConcurrentHashMap();
        }
        if (visibilityByCategory.isEmpty()) {
            setDefaultVisibilityForCategories();
        }
    }
    
}
