package org.lsst.ccs.command;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.command.RoutingCommandSet.RoutingDictionary;

/**
 * A set of utilities for dealing with command dictionaries.
 *
 * @author turri
 */
public abstract class DictionaryUtils {

    static final Map<DictionaryCommand,String> commandHelp = new HashMap<>();
    
    private DictionaryUtils() {
        
    }
    
    static String basicHelpForDictionary(Dictionary dict) {
        return basicHelpForDictionary(dict,new Options());
    }
    static String basicHelpForDictionary(Dictionary dict, Options opts) {
        return basicHelpForDictionary(dict, "", opts);
    }

    static String basicHelpForDictionary(Dictionary dict, String indent) {
        return basicHelpForDictionary(dict, indent, new Options());
    }
    static String basicHelpForDictionary(Dictionary dict, String indent, Options opts) {
        StringBuilder helpOut = new StringBuilder();
        List<DictionaryCommand> sorted = new ArrayList<>();
        sorted = fillListOfCommands(dict, sorted);
        Collections.sort(sorted, new CommandDefinitionComparator());
        DictionaryHelpGenerator helpGenerator = dict.getHelpGenerator();
        
        boolean showTargets = opts.hasOption("targets");
        
        for (DictionaryCommand def : sorted) {
            if ( ! showTargets && def instanceof RoutingCommandSet.RoutingCommand ) {
                continue;
            }
            
            String help = DictionaryUtils.basicHelpForCommand(def);
            
            if (helpGenerator != null && helpGenerator.hasHelp(def)) {
                help = helpGenerator.modifyHelpForCommand(def, help, true);
            }
            helpOut.append(indent).append(help);
        }
        return helpOut.toString();

    }
    
    private static List<DictionaryCommand> fillListOfCommands(Dictionary dict, List<DictionaryCommand> l) {
        if ( dict instanceof RoutingDictionary ) {
            if ( ((RoutingDictionary)dict).isRouteAvailable() && !((RoutingDictionary)dict).hasVisibleCommands() ) {
                return l;
            }
        }
        if(dict instanceof CompositeCommandDictionary) {
            CompositeCommandDictionary ccd = (CompositeCommandDictionary)dict;
            for (Dictionary d : ccd.getDictionaries()) {
                fillListOfCommands(d, l);
            }
        } else {
            for (DictionaryCommand def : dict.filterByVisibilityIterator()) {
                    l.add(def);
            }
        }
        return l;
    }

    static String basicHelpForCommand(DictionaryCommand def) {
        if ( commandHelp.containsKey(def) ) {
            return commandHelp.get(def);
        }
        StringBuilder builder = new StringBuilder();
        builder.append(def.getCommandName());
        for (DictionaryArgument param : def.getArguments()) {
            List<String> allowedValues = param.getAllowedValues();
            builder.append(' ').append(allowedValues.size() == 1 ? allowedValues.get(0) : param.getName());
        }
        if (def.isVarArgs()) {
            builder.append("...");
        }

        if (builder.length() <= 30) {
            for (int i = builder.length(); i < 31; i++) {
                builder.append(" ");
            }
        } else {
            builder.append("\n                               ");
        }
        builder.append(def.getDescription()).append("\n");

        if ( def.hasOptions() ) {
            builder.append("    supported options:\n");
            for (SupportedOption opt : def.getSupportedOptions()) {
                builder.append("                   --").append(opt.getName()).append(", -").append(opt.getSingleLetterName()).append("\t").append(opt.getDescription()).append("\n");
            }
            builder.append("\n");
        }
        
        if (def.getAliases().length > 0) {
            builder.append("    aliases:");
            for (String alias : def.getAliases()) {
                builder.append(" ").append(alias);
            }
            builder.append("\n");
        }
        String result = builder.toString();
        commandHelp.put(def,result);
        return result;
    }

    static DictionaryCommand findCommand(Dictionary dict, String command, int argCount) {
        try {
            return dict.findCommand(new BasicCommandNullArgs(command, argCount));
        } catch (CommandArgumentMatchException ex) {
            return null;
        }
    }

    static boolean containsCommand(Dictionary dict, String command, int argCount) {
        try {
            return dict.containsCommand(new BasicCommandNullArgs(command, argCount));
        } catch (CommandArgumentMatchException ex) {
            return false;
        }
    }

    /**
     * Utility method to check that if a DictionaryCommand is available in a
     * dictionary. This method is used primarily to detect ambiguous commands.
     *
     * The check proceeds as followd: -1- Check that there is a match with
     * either the name or one of the aliases -2- Check if they have the same
     * number of arguments -3- If they have allowed values, check that the sets
     * are disjoined -4- If default values are allowed, make sure the number of
     * parameters that can be passed don't overlap.
     *
     * @param dict The existing dictionary.
     * @param newCmd The new command.
     * @return false if the dictionary does not contain the provided dictionary
     * command.
     */
    static boolean containsDictionaryCommand(Dictionary dict, DictionaryCommand newCmd) {

        for (DictionaryCommand oldCommand : dict) {
            ArrayList<String> cmdNameOrAliases = new ArrayList<>();
            cmdNameOrAliases.add(oldCommand.getCommandName());
            String[] aliases = oldCommand.getAliases();
            if (aliases != null) {
                cmdNameOrAliases.addAll(Arrays.asList(aliases));
            }

            for (String cmdNameOrAlias : cmdNameOrAliases) {
                // -1- Check if there is any match with either the command name or an alias
                if (newCmd.getCommandName().equals(cmdNameOrAlias)) {

                    //When dealing with vararg methods we consider them ambigous when:
                    // -a- both of them are varargs
                    // or
                    // -b- the var arg method's number of arguments is less than the 
                    // other method's number of arguments + 1 to guarantee that
                    // the number of arguments alone can be used to tell them apart
                    if (newCmd.isVarArgs() && oldCommand.isVarArgs()) {
                        return true;
                    } else if (newCmd.isVarArgs() && (newCmd.getArguments().length - 1 <= oldCommand.getArguments().length)) {
                        return true;
                    } else if (oldCommand.isVarArgs() && (oldCommand.getArguments().length - 1 <= newCmd.getArguments().length)) {
                        return true;
                    }

                    if (newCmd.getArguments().length == oldCommand.getArguments().length) {
                        boolean isOk = false;
                        //If the commands have the same name and number of arguments,
                        //they can still be different if the arguments have disjoined
                        //sets of allowed values for their arguments
                        int nArgs = newCmd.getArguments().length;
                        for (int i = 0; i < nArgs; i++) {
                            DictionaryArgument newArg = newCmd.getArguments()[i];
                            DictionaryArgument oldArg = oldCommand.getArguments()[i];
                            if (newArg.getAllowedValues().isEmpty() || oldArg.getAllowedValues().isEmpty()) {
                                return true;
                            }
                            List<String> allowedValues = new ArrayList<>(newArg.getAllowedValues());
                            allowedValues.retainAll(oldArg.getAllowedValues());
                            if (!allowedValues.isEmpty()) {
                                return true;
                            } else {
                                isOk = true;
                                break;
                            }
                        }
                        if (isOk) {
                            continue;
                        }
                        return true;
                    }

                    //Also check that there are no ambiguities when arguments
                    //can be omitted because they have default values
                    //In this case the set of total arguments and minimum number
                    //of arguments that can be provided must be disjoined between
                    //the two commands.
                    int oldMaxArgs = oldCommand.getArguments().length;
                    int oldMinArgs = getDictionaryCommandMinArguments(oldCommand);
                    int newMaxArgs = newCmd.getArguments().length;
                    int newMinArgs = getDictionaryCommandMinArguments(newCmd);
                    if (newMinArgs <= oldMaxArgs && newMinArgs >= oldMinArgs) {
                        return true;
                    }
                    if (newMaxArgs <= oldMaxArgs && newMaxArgs >= oldMinArgs) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    private static int getDictionaryCommandMinArguments(DictionaryCommand dc) {
        int minArgs = 0;
        for (DictionaryArgument a : dc.getArguments()) {
            if (! a.hasDefaultValue()) {
                minArgs++;
            }
        }
        return minArgs;
    }

    /**
     * Test if the given command dictionary entry matches the given command.
     * This method takes into account the presence of default arguments,
     * varargs, and commands which take a specific enumerated set of legal
     * values.
     *
     * @param def The dictionary command entry
     * @param tc The command to test
     * @return <code>true</code> if the command matches the dictionary command
     * entry
     */
    static boolean commandMatch(DictionaryCommand def, BasicCommand tc) throws CommandArgumentMatchException {

        String command = tc.getCommand();
        int argumentCount = tc.getArgumentCount();

        ArrayList<String> cmdNameOrAliases = new ArrayList<>();
        cmdNameOrAliases.add(def.getCommandName());
        String[] aliases = def.getAliases();
        if (aliases != null) {
            cmdNameOrAliases.addAll(Arrays.asList(aliases));
        }
        for (String cmdNameOrAlias : cmdNameOrAliases) {
            //Check if there is any match with either the command name or an alias
            if (cmdNameOrAlias.equals(command)) {
                //Arguments matching:
                //   -> Same number of arguments
                //   -> More arguments with the extra ones with default values
                //   -> Dictionary command less arguments but has var args
                boolean argumentCountOk = false;
                int cmdNumberOfArguments = def.getArguments().length;
                if (argumentCount == cmdNumberOfArguments) {
                    argumentCountOk = true;
                } else if (argumentCount >= cmdNumberOfArguments - 1 && def.isVarArgs()) {
                    argumentCountOk = true;
                } else if (argumentCount < cmdNumberOfArguments) {
                    argumentCountOk = true;
                    for (int i = argumentCount; i < cmdNumberOfArguments; i++) {
                        if (!def.getArguments()[i].hasDefaultValue()) {
                            argumentCountOk = false;
                            break;
                        }
                    }
                }
                //If there is a match with the number of arguments
                //check if there is a match with the argument's value (if provided)
                //with the corresponding arguments allowed values.
                //If the match fails, false is returned;
                if (argumentCountOk) {
                    if (argumentCount == 0) {
                        return true;
                    }
                    Object[] argValues = tc.getArguments();
                    if (argValues != null) {
                        boolean argMatch = true;
                        for (int i = 0; i < argValues.length; i++) {
                            if (i >= def.getArguments().length) {
                                return true;
                            }
                            List<String> allowedValues = def.getArguments()[i].getAllowedValues();
                            if (allowedValues != null && !allowedValues.isEmpty()) {
                                argMatch = false;
                                for (String allowedValue : allowedValues) {
                                    if (allowedValue.toLowerCase().equals(argValues[i].toString().toLowerCase())) {
                                        argMatch = true;
                                        break;
                                    }
                                }
                            }
                            if (!argMatch) {
                                DictionaryArgument[] args = def.getArguments();
                                String[] names = new String[args.length];
                                for (int j=0; j<args.length; j++) {
                                    names[j] = args[j].getName();
                                }
                                throw new CommandArgumentMatchException(tc.getCommand(), i, names, String.valueOf(argValues[i]), allowedValues);
                            }
                        }
                        return true;
                    } else {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    private static class BasicCommandNullArgs implements BasicCommand {

        /**
         *
         */
        private static final long serialVersionUID = -744230510544974099L;
        private final String commandName;
        private final int argumentCount;

        BasicCommandNullArgs(String commandName, int argumentCount) {
            this.commandName = commandName;
            this.argumentCount = argumentCount;
        }

        @Override
        public int getArgumentCount() {
            return argumentCount;
        }

        @Override
        public String getCommand() {
            return commandName;
        }

        @Override
        public Object getArgument(int i) {
            return null;
        }

        @Override
        public Object[] getArguments() {
            return null;
        }

        @Override
        public String toString() {
            return prettyToString();
        }

        @Override
        public Options getOptions() {
            return new Options();
        }

    }

    /**
     * A comparator used for putting commands into alphabetical order
     */
    private static class CommandDefinitionComparator implements Comparator<DictionaryCommand> {

        @Override
        public int compare(DictionaryCommand o1, DictionaryCommand o2) {
            return o1.getCommandName().compareTo(o2.getCommandName());
        }
    }

    public static boolean areDictionariesEqual(Dictionary d1, Dictionary d2) {
        if (d1.getClass() != d2.getClass()) {
            return false;
        }
        if (d1.size() != d2.size()) {
            return false;
        }
        for (DictionaryCommand dc1 : d1) {
            DictionaryCommand dc2 = findCommand(d2, dc1.getCommandName(), dc1.getArguments().length);
            if ( dc2 == null ) {
                return false;
            }
            if ( ! areDictionaryCommandsEqual(dc1, dc2) ) {
                return false;
            }
        }

        return true;
    }

    /**
         * Check if two dictionary commands are equal.
     *
     * @param dc1
     * @param dc2
     * @return true if they are equal, false otherwise.
     */
    public static boolean areDictionaryCommandsEqual(DictionaryCommand dc1, DictionaryCommand dc2) {
        if (dc2.getCommandName().equals(dc1.getCommandName())) {
            if (dc2.isAutoAck() == dc1.isAutoAck() && dc2.isVarArgs() == dc1.isVarArgs()) {
                if (Arrays.equals(dc2.getAliases(), dc1.getAliases())) {
                    if (dc2.getCategory() == dc1.getCategory() && (dc2.getDescription() == null ? dc1.getDescription() == null : dc2.getDescription().equals(dc1.getDescription()))) {
                        if (dc2.getLevel() == dc1.getLevel() && dc2.getTimeout() == dc1.getTimeout() && dc2.getType() == dc1.getType()) {
                            return areDictionaryArgumentsEqual(dc1.getArguments(), dc2.getArguments());
                        }
                    }
                }
            }
        }
        return false;
    }

    /**
     * Check if two arrays of DictionaryArguments are equals. The arguments are
     * expected to be in the same order within the array.
     *
     * @param da1s
     * @param da2s
     * @return true if they are equal.
     */
    public static boolean areDictionaryArgumentsEqual(DictionaryArgument[] da1s, DictionaryArgument[] da2s) {
        if (da2s.length == da1s.length) {
            for (int i = 0; i < da2s.length; i++) {
                DictionaryArgument da2 = da2s[i];
                DictionaryArgument da1 = da1s[i];
                if (!da2.hasDefaultValue() && da1.hasDefaultValue() ) {
                    return false;
                }
                if (da2.getDefaultValue() == null ? da1.getDefaultValue() != null : !da2.getDefaultValue().equals(da1.getDefaultValue())) {
                    return false;
                }
                       
                if (!da2.getDescription().equals(da1.getDescription())) {
                    return false;
                }
                if (!da2.getName().equals(da1.getName())) {
                    return false;
                }
                if (!da2.getSimpleType().equals(da1.getSimpleType())) {
                    return false;
                }
                if (!da2.getType().equals(da1.getType())) {
                    return false;
                }
                if (!da2.getAllowedValues().equals(da1.getAllowedValues())) {
                    return false;
                }
            }
        }
        return true;
    }
}
