package org.lsst.ccs.command;

import java.util.*;
import java.util.stream.Collectors;
import org.lsst.ccs.utilities.misc.Similarity;

/**
 * An exception thrown when a command Argument does not match the provided values.
 * 
 */
public class CommandArgumentMatchException extends Exception {
    
    static private int MAX = 10; // maximum number of allowed values to display
    static private int MIN = 5; // minimum number of allowed values to hide
    
    private final String command;
    private final int index;
    private final String[] names;
    private final String value;
    private final ArrayList<String> allowedValues;
    private final LinkedList<CommandArgumentMatchException> children;

    @Deprecated
    public CommandArgumentMatchException(String message) {
        throw new UnsupportedOperationException();
    }

    public CommandArgumentMatchException(String command, int index, String[] names, String value, List<String> allowedValues) {
        this.command = command;
        this.index = index;
        this.names = names;
        this.value = value;
        this.allowedValues = new ArrayList<>(new LinkedHashSet<>(allowedValues)); // remove duplicates
        this.children = null;
    }

    private CommandArgumentMatchException(LinkedList<CommandArgumentMatchException> children) {
        this.command = null;
        this.index = 0;
        this.names = null;
        this.value = null;
        this.allowedValues = null;
        this.children = children;
    }
        
    protected static void throwExceptionIfNeeded(List<CommandArgumentMatchException> exceptions) throws CommandArgumentMatchException {
        if (exceptions.size() == 1) {
            throw exceptions.get(0);
        } else if (exceptions.size() > 1) {
            LinkedList<CommandArgumentMatchException> xx = new LinkedList<>();
            exceptions.stream().flatMap(e -> e.getConstituents().stream()).forEach(e -> {
                boolean notDone = true;
                ListIterator<CommandArgumentMatchException> it = xx.listIterator();
                while (notDone && it.hasNext()) {
                    CommandArgumentMatchException x = it.next();
                    if (x.index < e.index) {
                        it.previous();
                        it.add(e);
                        notDone = false;
                    } else {
                        if (x.command.equals(e.command) && x.index == e.index && Arrays.equals(x.names, e.names)) {
                            LinkedHashSet<String> aw = new LinkedHashSet<>(e.allowedValues);
                            aw.removeAll(x.allowedValues);
                            x.allowedValues.addAll(aw);
                            notDone = false;
                        }
                    }
                }
                if (notDone) {
                    xx.add(e);
                }
            });
            throw xx.size() == 1 ? xx.get(0) : new CommandArgumentMatchException(xx);
        }
    }

    @Override
    public String getMessage() {
        if (children == null) {
            StringBuilder sb = new StringBuilder();
            sb.append("illegal value \"").append(value).append("\" for argument \"").append(names[index]);
            sb.append("\" in \"").append(command);
            for (String name : names) {
                sb.append(" ").append(name);
            }
            sb.append("\".");
            sb.append(" Allowed values: ").append(truncateAllowedValues(value, allowedValues)).append(".");
            return sb.toString();
        } else {
            return String.join("\nOr ", children.stream().map(x -> x.getMessage()).collect(Collectors.toList()));
        }
    }
    
    private String truncateAllowedValues(String value, List<String> allowedValues) {
        if (allowedValues.size() > MAX) {
            List<String> values = Similarity.head(value, allowedValues, MAX, Integer.MAX_VALUE);
            int hidden = allowedValues.size() - values.size();
            if (hidden >= MIN) {
                return values.toString() +" and "+ hidden +" more";
            }
        }
        return allowedValues.toString();
    }
    
    private List<CommandArgumentMatchException> getConstituents() {
        if (children == null) {
            return Collections.singletonList(this);
        } else {
            LinkedList<CommandArgumentMatchException> out = new LinkedList<>();
            children.forEach(x -> out.addAll(x.getConstituents()));
            return out;
        }
    }

}
