package org.lsst.ccs.command;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

/**
 * Performs command completion, based on the information available in the
 * command dictionary. This is designed to be jline compatible, but not actually
 * implement jline's command completer to avoid adding a permanent dependency on
 * jline.
 *
 * @author tonyj
 */
class DefaultDictionaryCompleter implements DictionaryCompleter, Serializable {

    private static final long serialVersionUID = 8413764867907456999L;
    private final Dictionary dict;

    /**
     * Create a command completer for the given dictionary.
     *
     * @param dict
     */
    public DefaultDictionaryCompleter(Dictionary dict) {
        this.dict = dict;
    }

    /**
     * Generate a list of possible command completions.
     *
     * @param buffer The current content of the input line
     * @param index The index within the input line of the cursor
     * @param list A list which this routine should fill with possible command
     * completions.
     * @return The position at which the cursor should be positioned prior to
     * adding any of the returned items to the buffer.
     */
    @Override
    public int complete(String buffer, int index, List<CharSequence> list) {
        TokenizedCommand tc = new TokenizedCommand(buffer.substring(0, index));
        int argCount = tc.getArgumentCount();
        boolean endsWithWhiteSpace = !tc.isEmpty() && Character.isWhitespace(buffer.charAt(index - 1));
        //If the command line ends with a white space, completion is being sought for
        //an additional argument. For this reason we increase the count by one.
        if (endsWithWhiteSpace) {
            argCount++;
        }
        //The following test is to address a bug introduced by declaring the
        //dict field transient. This check is only temporary and can be
        //removed at a later time.
        if ( dict == null ) {
            return index;
        }
        
        boolean isCompletingOption = tc.getLastToken() != null && tc.getLastToken().isOption();
        if ( isCompletingOption ) {
            boolean isShortOption = tc.getLastToken().isShortOption();
            String startOpt = tc.getLastToken().getString();
            List<List<SupportedOption>> supportedOptions = new ArrayList<>();
            String commandName = tc.getCommand();
            for (DictionaryCommand def : dict.filterByVisibilityIterator()) {
                String command = def.getCommandName();
                if ( !command.equals(commandName) ) {
                    continue;
                }
                if ( !def.hasOptions() ) {
                    continue;
                }
                if ( def.getArguments().length < argCount && ! def.isVarArgs() ) {
                    continue;
                }
                supportedOptions.add(def.getSupportedOptions());
            }
            if ( supportedOptions.size() == 1 ) {
                List<SupportedOption> opts = supportedOptions.get(0);
                for ( SupportedOption opt : opts) {
                    if (isShortOption) {
                        if (!tc.getOptions().hasOption(opt.getName()) && !tc.getOptions().hasOption(opt.getSingleLetterName())) {
                            list.add(opt.getSingleLetterName() + " ");
                        }
                    } else {
                        if (!startOpt.isEmpty() && opt.getName().startsWith(startOpt) ) {
                            list.add(opt.getName() + " ");
                        } else if (startOpt.isEmpty() && !tc.getOptions().hasOption(opt.getName()) && !tc.getOptions().hasOption(opt.getSingleLetterName())) {
                            list.add(opt.getName() + " ");                            
                        }
                    }
                }
            }
                        
            if ( isShortOption ) {
                return index;
            }
            return tc.getLastToken().getLocation();

        }
        
        
        if (tc.isEmpty()) {
            SortedSet<CharSequence> set = new TreeSet<>();
            for (DictionaryCommand def : dict.filterByVisibilityIterator()) {
                    String command = def.getCommandName();
                    command += " ";
                    set.add(command);
                    for (String alias : def.getAliases()) {
                        alias += " ";
                        set.add(alias);
                    }
            }

            list.addAll(set);
        } else if (argCount == 0) {
            String prefix = tc.getCommand();
            SortedSet<CharSequence> set = new TreeSet<>();
            Set<DictionaryCommand> uniqueCommands = new HashSet<>();
            for (DictionaryCommand def : dict.filterByVisibilityIterator()) {
                    String command = def.getCommandName();
                    if (command.startsWith(prefix)) {
                        command += " ";
                        set.add(command);
                        uniqueCommands.add(def);
                    }
                    for (String alias : def.getAliases()) {
                        if (alias.startsWith(prefix)) {
                            alias += " ";
                            set.add(alias);
                            uniqueCommands.add(def);
                        }
                    }
            }
            // Special case, if there are >1 matching commands, but they are all just
            // aliases for the same command, only return the actual command
            if (set.size() > 1 && uniqueCommands.size() == 1) {
                list.add(uniqueCommands.iterator().next().getCommandName() + " ");
            } else {
                list.addAll(set);
            }
            return tc.getCommandLocation();
        } else {
            String command = tc.getCommand();

            String lastArg = endsWithWhiteSpace ? "" : tc.getArgument(argCount - 1);
            
            // This is used if there are no commands with matching default values.
            List<String> defaultCompletions = new ArrayList<>();
            boolean useDefaultCompletions = true;

            for (DictionaryCommand def : dict.filterByVisibilityIterator()) {
                    if (def.matchesCommandOrAlias(command) && (def.getArguments().length >= argCount || def.isVarArgs())) {                    
                        int currentArgIndex = Math.min(argCount, def.getArguments().length) - 1;                                        
                        final DictionaryArgument currentArg = def.getArguments()[currentArgIndex];
                        
                        //Check if the given argument has a match for the previously provided arguments
                        //If not the current command has to be skipped.
                        boolean skipCurrentCommand = false;
                        for (int i = currentArgIndex - 1; i >= 0; i--) {
                            DictionaryArgument previousArg = def.getArguments()[i];
                            if (previousArg.getAllowedValues().size() > 0  && !previousArg.getAllowedValues().contains(tc.getArgument(i))) {
                                skipCurrentCommand = true;
                            }
                        }
                        if (skipCurrentCommand) {
                            continue;
                        }
                        
                        List<String> values = currentArg.getAllowedValues(); 
                        if (values.isEmpty()) {
                            // The argument is not an enumeration, return argument description, prefixed with # 
                            String helpStr = "# " + currentArg.getName() + " " + currentArg.getSimpleType() + " " + currentArg.getDescription();
                            if ( ! defaultCompletions.contains(helpStr) ) {
                                defaultCompletions.add(helpStr);
                            }
                        } else {
                            useDefaultCompletions = false;
                            // The argument is an enumeration, list possible values    
                            for (String value : values) {
                                if (startsWithIgnoreCase(value, lastArg)) {
                                    value += " ";
                                    if ( ! list.contains(value) ) {
                                        list.add(value);
                                    }
                                }
                            }
                        }
                    }
            }
            if (useDefaultCompletions) {
                for (String completion : defaultCompletions) {
                    list.add(completion);
                }
            }
            return endsWithWhiteSpace ? index : tc.getArgumentLocation(argCount - 1);
        }
        return index;
    }

    private static boolean startsWithIgnoreCase(String reference, String test) {
        if (test.length() > reference.length()) {
            return false;
        }
        return reference.substring(0, test.length()).equalsIgnoreCase(test);
    }

}
