package org.lsst.ccs.command;

import java.io.IOException;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.lsst.ccs.command.annotations.Option;

/**
 * An implementation of DictionaryCommand based on a single annotated method.
 *
 * @author turri
 */
class MethodBasedDictionaryCommand implements DictionaryCommand {

    private final String description;
    private final String[] aliases;
    private final DictionaryArgument[] params;
    private final Command.CommandType type;
    private transient Command.CommandCategory category;
    private final String name;
    private final boolean hasVarArgs;
    private final int level;
    private final boolean autoAck;
    private final Duration timeout;
    private static final long serialVersionUID = -2134147357955297580L;
    private List<SupportedOption> supportedOptions = new ArrayList<>();
    
    /**
     * Create a command definition from a method and associated annotation.
     *
     * @param annotatedMethod The method providing the command implementation
     * @param annotation The annotation on the method
     */
    MethodBasedDictionaryCommand(Method annotatedMethod, Command annotation) {
        
        this.description = annotation.description();
        this.aliases = splitAliases(annotation.alias());
        this.type = annotation.type();
        this.category = annotation.category();
        this.name = annotation.name().isEmpty() ? annotatedMethod.getName() : annotation.name();
        this.hasVarArgs = annotatedMethod.isVarArgs();
        this.level = annotation.level();
        this.autoAck = annotation.autoAck();
        this.timeout = annotation.timeout() > 0 ? Duration.ofSeconds(annotation.timeout()) : null;
        
        //Does this method support Options?
        Option[] options = annotatedMethod.getAnnotationsByType(Option.class);        
        Map<String, SupportedOption> shortNamesForMethod = new HashMap<>();
        for (Option o : options) {
            SupportedOption so = new SupportedOption(o.name(), o.description(), o.singleLetterName());
            supportedOptions.add(so);
            String singleLetter = so .getSingleLetterName();
            SupportedOption otherSo = shortNamesForMethod.get(singleLetter);
            if ( otherSo != null ) {
                throw new IllegalArgumentException("Method "+annotatedMethod.getName()+" contains at least two options with single letter name = "+singleLetter+" : "+so.getName()+" "+otherSo.getName());
            }
            shortNamesForMethod.put(singleLetter, so);
        }
        boolean hasOptions = !supportedOptions.isEmpty();
        
        
        Class[] types = annotatedMethod.getParameterTypes();
        Annotation[][] parAnnotations = annotatedMethod.getParameterAnnotations();

        
        
        params = new MethodBasedDictionaryArgument[ hasOptions ? types.length -1 : types.length ];
        Parameter[] methodParameters = annotatedMethod.getParameters();
        
        boolean foundOptionsArgument = false;
        int argCount = 0;
        
        for (int i = 0; i < types.length; i++) {

            String parName = methodParameters[i].getName();
            String parDescription = "";
            String defaultValue = Argument.NOT_SET;
            Method allowedValueMethod = null;
            for (Annotation a : parAnnotations[i]) {
                if (a instanceof Argument) {
                    Argument paramAnnotation = (Argument) a;
                    parName = paramAnnotation.name().isEmpty() ? parName : paramAnnotation.name();
                    parDescription = paramAnnotation.description();
                    if (!paramAnnotation.defaultValue().equals(Argument.NOT_SET)) {
                        defaultValue = paramAnnotation.defaultValue();
                    }
                    if (!paramAnnotation.allowedValueProvider().isEmpty()) {
                        try {
                            allowedValueMethod = annotatedMethod.getDeclaringClass().getMethod(paramAnnotation.allowedValueProvider());
                        } catch (Exception ex) {
                            // ignore silently
                        }
                    }
                    break;
                }
            }
            Class parameterType = hasVarArgs && i == types.length - 1 ? types[i].getComponentType() : types[i];
            if ( Options.class.isAssignableFrom(parameterType) ) {
                if (!hasOptions) {
                    throw new RuntimeException("Method "+annotatedMethod.getDeclaringClass().getSimpleName()+"::"+name+" has an Options argument but no @Option annotations. Please annotate the method accordingly or remove the Options argument.");
                }
                if ( foundOptionsArgument ) {
                    throw new RuntimeException("Method "+annotatedMethod.getDeclaringClass().getSimpleName()+"::"+name+" has one or more @Option annotations and more than one Options argument. There can be only one.");
                }
                foundOptionsArgument = true;                                
                if ( foundOptionsArgument && i != 0 ) {
                    throw new RuntimeException("Method "+annotatedMethod.getDeclaringClass().getSimpleName()+"::"+name+" supports Options but the Options argument is not the first one in the calling sequence.");
                }
            } else {
                params[argCount++] = new MethodBasedDictionaryArgument(parName, parameterType, parDescription, defaultValue, allowedValueMethod);
            }
            
        }

    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public String[] getAliases() {
        return aliases;
    }

    @Override
    public DictionaryArgument[] getArguments() {
        return params;
    }

    @Override
    public Command.CommandType getType() {
        return type;
    }

    @Override
    public Command.CommandCategory getCategory() {
        return category;
    }

    @Override
    public String getCommandName() {
        return name;
    }

    @Override
    public boolean isVarArgs() {
        return hasVarArgs;
    }

    @Override
    public int getLevel() {
        return level;
    }

    @Override
    public boolean isAutoAck() {
        return autoAck;
    }

    @Override
    public Duration getTimeout() {
        return timeout;
    }

    private String[] splitAliases(String alias) {
        return alias.length() > 0 ? alias.split("\\s?,\\s?") : NO_ALIASES;
    }

    @Override
    public List<SupportedOption> getSupportedOptions() {
        //This is needed to support deserialization from older
        //versions that still don't support SupportedOption.
        if ( supportedOptions == null ) {
            return new ArrayList<>();
        }
        return supportedOptions;
    }
    
    

    @Override
    public String toString() {
        return "MethodBasedDictionaryCommand{"
                + "description='" + description + '\''
                + ", aliases=" + Arrays.toString(aliases)
                + ", params=" + Arrays.toString(params)
                + ", type=" + type
                + ", name='" + name + '\''
                + ", hasVarArgs=" + hasVarArgs
                + '}';
    }
    
    
    
    /**
     * We had to add custom serialization due to having added the CORE Command
     * Category. These methods can be removed the next time we have a backward
     * incompatible release.
     */
    private void writeObject(java.io.ObjectOutputStream stream)
            throws IOException {
        stream.defaultWriteObject();
        stream.writeObject(category.name());
    }

    private void readObject(java.io.ObjectInputStream stream)
            throws IOException, ClassNotFoundException {
        stream.defaultReadObject();
        try {
            String categoryName = (String)stream.readObject();
            Command.CommandCategory tmpCategory = Command.CommandCategory.valueOf(categoryName);
            category = tmpCategory;
        } catch (Exception e) {
            //Silently ignore.        
        } finally {
            if ( category == null ) {
                category = Command.CommandCategory.USER;
            }
        }
    }    
}
