package org.lsst.ccs.command;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import org.lsst.ccs.command.RoutingCommandSet.RoutingDictionary;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;

/**
 * A CommandSet that acts as a filter and deals with changing the active route.
 *
 * @author turri
 */
public class RouteSelectionCommandSet extends CompositeCommandSet {

    private String selectedRoute = "";
    private final CompositeCommandSet activeRouteCommandSet = new CompositeCommandSet();
    
    // RoutingCommandSets stored by path
    private final Map<String, RoutingCommandSet> paths = new ConcurrentHashMap<>();
    
    public RouteSelectionCommandSet() {
        CommandSetBuilder builder = new CommandSetBuilder();
        final BuiltIns builtIns = new BuiltIns();
        final CommandSet builtInCommandSet = builder.buildCommandSet(builtIns);
        add(builtInCommandSet);
        add(activeRouteCommandSet);
    }
    public RouteSelectionCommandSet(CompositeCommandSet parentCommandSet) {
        this();
        add(parentCommandSet);
    }

    private void updateActiveRoute() {
        activeRouteCommandSet.clear();
        if (!"".equals(selectedRoute)) {
            Map<String, RoutingCommandSet> routesMap = getAvailableRoutesMap(this, false);
            for (Entry<String, RoutingCommandSet> entry : routesMap.entrySet()) {
                String routeName = entry.getKey();
                if (routeName.equals(selectedRoute)) {
                    CommandSet c = entry.getValue().getRouteCommandSet();
                    activeRouteCommandSet.add(c);
                } else if (routeName.startsWith(selectedRoute + "/")) {
                    String newRoute = routeName.replace(selectedRoute + "/", "");
                    CommandSet c = new RoutingCommandSet(newRoute, entry.getValue().getPath(), entry.getValue().getRouteCommandSet());
                    activeRouteCommandSet.add(c);
                }
            }
        }
    }

    void setActiveRoute(String value) {
        if ("".equals(value)) {
            selectedRoute = "";
        } else {
            Map<String, RoutingCommandSet> rs = getAvailableRoutesMap(this, false);
            //The provided value can either be full or partial route.
            //If the provided value is not a valid route it might be partial
            String fullRoute = !rs.containsKey(value) && !"".equals(selectedRoute) ? selectedRoute + "/" + value : value;
            if (!rs.containsKey(fullRoute)) {
                throw new RuntimeException("Error: could not find target \"" + value + "\".");
            }
            selectedRoute = fullRoute;
        }
        updateActiveRoute();
    }

    String getActiveRoute() {
        return selectedRoute;
    }

    /**
     * Set the level for the given route and all the sub targets defined by this
     * route.
     *
     * @param route
     * @param level
     */
    public void setRouteLevel(String route, int level) {
        Map<String, RoutingCommandSet> routes = getAvailableRoutesMap(this, false);

        String fullRoute = !routes.containsKey(route) && !"".equals(selectedRoute) ? selectedRoute + "/" + route : route;

        if (!routes.containsKey(fullRoute)) {
            throw new RuntimeException("Error: could not find target \"" + route + "\".");
        }

        for (Map.Entry<String, RoutingCommandSet> e : routes.entrySet()) {
            String r = e.getKey();
            if (fullRoute.equals(r) || r.startsWith(fullRoute + "/")) {
                e.getValue().getCommandDictionary().setLevel(Integer.valueOf(level));
            }
        }
    }

    public int getRouteLevel(String route) {
        Map<String, RoutingCommandSet> routes = getAvailableRoutesMap(this, false);

        String fullRoute = !routes.containsKey(route) && !"".equals(selectedRoute) ? selectedRoute + "/" + route : route;

        if (!routes.containsKey(fullRoute)) {
            throw new RuntimeException("Error: could not find target \"" + route + "\".");
        }
        return routes.get(fullRoute).getCommandDictionary().getLevel();
    }

    public void setRouteCommandTypeVisibility(String route, Command.CommandType... types) {
        Map<String, RoutingCommandSet> routes = getAvailableRoutesMap(this, false);

        String fullRoute = !routes.containsKey(route) && !"".equals(selectedRoute) ? selectedRoute + "/" + route : route;

        if (!routes.containsKey(fullRoute)) {
            throw new RuntimeException("Error: could not find target \"" + route + "\".");
        }

        for (Map.Entry<String, RoutingCommandSet> e : routes.entrySet()) {
            String r = e.getKey();
            if (fullRoute.equals(r) || r.startsWith(fullRoute + "/")) {
                e.getValue().getCommandDictionary().setVisibilityForTypes(types);
            }
        }
    }
    
    /**
     * Add a command set at the specified path. This internally creates a
     * RoutingCommandSet, the route of which is specified by the route parameter.
     *
     * WHAT IS THE DIFFERENCE BETWEEN PATH AND ROUTE?
     * 
     * @param route the route corresponding to the CommandSet
     * @param path the command set path
     * @param parent the parent command set
     */
    public void addRoutingCommandSet(String route, String path, CommandSet parent) {
        if (!paths.containsKey(path)) {
            RoutingCommandSet crcs = new RoutingCommandSet(route, path, parent);
            add(crcs);
            paths.put(path, crcs);
        } else {
            throw new RuntimeException("Error: Route " + path + " already exists");
        }
    }
    
    /**
     * Remove all paths that start with or are equal to {@code route}.
     * @param route the root route to remove.
     */
    public void removeRoute(String route) {
        Iterator<Map.Entry<String, RoutingCommandSet>> it = paths.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<String, RoutingCommandSet> e = it.next();
            String routeName = e.getKey();
            if (routeName.equals(route) || routeName.startsWith(route + "/")) {
                remove(e.getValue());
                    it.remove();
            }
        }
    }
    
    /**
     * 
     * @param path
     * @return 
     */
    public boolean containsPath(String path) {
        return paths.containsKey(path);
    }
    
    /**
     * Return a list of routes and their associated {@code RoutingCommandSet}
     * @return 
     */
    public Map<String, RoutingCommandSet> getRoutes() {
        Map<String, RoutingCommandSet> res = new LinkedHashMap<>();
            for (Map.Entry<String, RoutingCommandSet> entry : paths.entrySet()) {
                res.put(entry.getValue().getRoute(), entry.getValue());
        }
        return res;
    }
    
    /**
     * Return all the command sets below {@code root}
     *
     * @param root the root route.
     * @return a Map of CommandSets. Note that the values are not
     * RoutingCommandSet objects, but their corresponding inner CommandSet
     */
    public Map<String, CommandSet> getCommandSetsForRoute(String root) {
       Map<String, CommandSet> res = new HashMap<>();
        for (Map.Entry<String, RoutingCommandSet> entry : paths.entrySet()) {
            String path = entry.getKey();
            if(path.equals(root) || path.startsWith(root+"/")) {
                res.put(path, entry.getValue().getRouteCommandSet());
            }
        }
        return res;
    }
    
    public enum SetGetTargetCommands {
        TARGET
    };

    /* This must be public for the command invocation to work */
    public class BuiltIns {

        @Command(description = "Set the default command target", level = Command.NORMAL)
        public void set(@Argument(name = "item") SetGetTargetCommands what, @Argument(name = "value", defaultValue = "", allowedValueProvider = "getSetAllowedValues") String value) {
            switch (what) {
                case TARGET:
                    setActiveRoute(value);
            }
        }

        public List<String> getSetAllowedValues() {
            updateActiveRoute();
            Map<String, RoutingCommandSet> availableRouteMaps = getAvailableRoutesMap(RouteSelectionCommandSet.this, true);
            
            //Then add the allowed values from the active paths.
            for (CommandSet cs : activeRouteCommandSet.getCommandSets()) {
                if (cs instanceof RoutingCommandSet) {
                    RoutingCommandSet rcs = (RoutingCommandSet)cs;
                    if(availableRouteMaps.containsKey(selectedRoute+"/"+rcs.getRoute())) {
                        availableRouteMaps.put(rcs.getRoute(), rcs);
                    }
                }
            }
            return new ArrayList<>(availableRouteMaps.keySet());
        }

        @Command(description = "Get various settings", level = Command.NORMAL)
        public String get(@Argument(name = "item") SetGetTargetCommands what) {
            switch (what) {
                case TARGET:
                    return getActiveRoute();
                default:
                    return null;
            }
        }

    }

    private static Map<String, RoutingCommandSet> getAvailableRoutesMap(CommandSet cs, boolean filterByLevel) {
        Map<String, RoutingCommandSet> result = new LinkedHashMap<>();
        if (cs instanceof RoutingCommandSet) {
            RoutingCommandSet rcs = ((RoutingCommandSet) cs);
            String fullRoute = rcs.getRoute();
            int currLevel = rcs.getCommandDictionary().getLevel();
            DictionaryCommand dc = ((RoutingDictionary) rcs.getCommandDictionary()).getRouteCommand(fullRoute);
            if (!filterByLevel || dc.getLevel() <= currLevel) {
                result.put(fullRoute, (RoutingCommandSet) cs);
            }
        } else if (cs instanceof CompositeCommandSet) {
            for (CommandSet s : ((CompositeCommandSet) cs).getCommandSets()) {
                result.putAll(getAvailableRoutesMap(s, filterByLevel));
            }
        }
        return result;
    }

}
