package org.lsst.ccs.command;

import java.io.Serializable;
import java.util.Iterator;
import java.util.function.Predicate;
import org.lsst.ccs.command.annotations.Command;

/**
 * A command dictionary contains all the information needed to provide
 * help and perform tab completion. It does not by itself provide sufficient
 * functionality to invoke a command, for this a CommandSet which is a combination of a 
 * command dictionary and a command invoker is required. A CommandDictionary is a collection of
 * CommandDefinitions, plus a few convenience methods.
 * @author tonyj
 */
public interface Dictionary extends Iterable<DictionaryCommand>, Serializable {

    /**
     * Test if a given command is present in a dictionary.
     * @param command The command (or alias) to search for
     * @return <code>true</code>If the command is found
     * @throws org.lsst.ccs.command.CommandArgumentMatchException if there is an illegal argument.
     */
    public boolean containsCommand(BasicCommand command) throws CommandArgumentMatchException;
    
    /**
     * Find a given command in the dictionary
     * @param command The command (or alias) to search for
     * @return The DictionaryCommand, or <code>null</code> if 
     * the command is not found.
     * @throws org.lsst.ccs.command.CommandArgumentMatchException if there is an illegal argument.
     */
    public DictionaryCommand findCommand(BasicCommand command) throws CommandArgumentMatchException;

    /**
     * The size of the dictionary.
     * @return The number of commands in this dictionary.
     */
    public int size();  
        
    /**
     * Get the DictionaryHelpGenerator for this Dictionary.
     * 
     * @return The DictionaryHelpGenerator. null if none is available.
     */
    default DictionaryHelpGenerator getHelpGenerator() {
        return null;
    }
    
    /**
     * Get the DictionaryCompleter for this Dictionary.
     * 
     * @return The DictionaryCompleter for this Dictionary.
     */
    default DictionaryCompleter getDictionaryCompleter() {
        return null;
    }
    
    /**
     * Set the visibility level for this dictionary.
     * A negative level will consider level 0 QUERY commands only.
     * A positive level will consider all types of commands with level up to {@code level}.
     * @param level the level to set.
     */
    void setLevel(int level);
    
    /**
     * Get the visibility level for this dictionary.
     * @return the level
     */
    int getLevel();
    
    /**
     * Specified command types will be visible, other ones will not be visible
     * @param types a list of types.
     */
    void setVisibilityForTypes(Command.CommandType ... types);
    
    /**
     * Get the type related visibility of a command that belongs to this dictionary.
     * @param type the command type to get the visibility for.
     * @return true if this command type is visible for this dictionary. 
     */
    boolean getVisibilityForType(Command.CommandType type);
    
    /**
     * An iterator over DictionaryCommand of this dictionary that filters by level.
     * @return 
     */
    default Iterable<DictionaryCommand> filterByLevelIterator() {
        return new Iterable<DictionaryCommand>() {
            
            @Override
            public Iterator<DictionaryCommand> iterator() {
                
                Iterator<DictionaryCommand> filteredIterator = new Iterator<DictionaryCommand>() {
                    
                    private final Iterator<DictionaryCommand> superIterator = Dictionary.this.iterator();
                    private boolean hasNext;
                    private DictionaryCommand nextDC = null;
                    
                    // A predicate that accepts commands that should be visible
                    private final Predicate<DictionaryCommand> pred = (dc) -> {
                        
                        return dc.getLevel() <= getLevel() && getVisibilityForType(dc.getType());
                    };
                    
                    @Override
                    public boolean hasNext() {
                        return hasNext;
                    }
                    
                    @Override
                    public DictionaryCommand next() {
                        DictionaryCommand prevDC = nextDC;
                        while (superIterator.hasNext()) {
                            DictionaryCommand dc = superIterator.next();
                            if (pred.test(dc)) {
                                hasNext = true;
                                nextDC = dc;
                                return prevDC;
                            }
                        }
                        hasNext = false;
                        return prevDC;
                    }
                };
                filteredIterator.next();
                return filteredIterator;
            }
        };
    }
}
