package org.lsst.ccs.command;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

/**
 * A command dictionary to which other command dictionaries can be added and
 * removed.
 *
 * @author tonyj
 */
/*
 * note by Bernard AMADE: why was this class turned public though it was package friendly?
 * if you implement a method f() that returns a Dictionary
 * this method is built using a CommandSetBuilder
 * now if you have another method g() with the same feature
 * if you want to create a composite which uses the result of f() and g()
 * you have to reuse 2 CommandSetBuilder from the ground up .
 * or  have methods m() and n() that return an intermediary CommandSet
 * and build first a CompositeCommandSet!
 * so not worth the hassle!
 */
public class CompositeCommandDictionary implements Dictionary {

    /**
	 * 
	 */
	private static final long serialVersionUID = 6746559441105266344L;
	private final LinkedHashSet<Dictionary> dicts = new LinkedHashSet<>();
    private final CompositeCommandDictionaryCompleter completer = new CompositeCommandDictionaryCompleter(dicts);

    public void add(Dictionary commandDictionary) {
        dicts.add(commandDictionary);
    }

    public void remove(Dictionary commandDictionary) {
        dicts.remove(commandDictionary);
    }

    void clear() {
        dicts.clear();
    }

    @Override
    public DictionaryCompleter getDictionaryCompleter() {
        return completer;
    }

    @Override
    public boolean containsCommand(BasicCommand tc) throws CommandArgumentMatchException {
        List<CommandArgumentMatchException> exceptions = new ArrayList<>();
        for (Dictionary dict : dicts) {
            try {
                if (dict.containsCommand(tc)) {
                    return true;
                }
            } catch (CommandArgumentMatchException ex) {
                exceptions.add(ex);
            }
        }
        CommandArgumentMatchException.throwExceptionIfNeeded(exceptions);
        return false;
    }

    @Override
    public DictionaryCommand findCommand(BasicCommand tc) throws CommandArgumentMatchException {
        List<DictionaryCommand> matches = new ArrayList<>();
        List<CommandArgumentMatchException> exceptions = new ArrayList<>();
        for (Dictionary dict : dicts) {
            try {
                if (dict.containsCommand(tc)) {
                    matches.add(dict.findCommand(tc));
                }
            } catch (CommandArgumentMatchException ex) {
                exceptions.add(ex);
            }
        }
        if (matches.isEmpty()) {
            CommandArgumentMatchException.throwExceptionIfNeeded(exceptions);
            return null;
        } else if (matches.size() == 1) {
            return matches.get(0);
        } else {
            throw new AmbiguousCommandException("Found " + matches.size() + " matches when looking for command " + tc.getCommand() + " with " + tc.getArgumentCount() + " arguments");
        }
    }

    @Override
    public Iterator<DictionaryCommand> iterator() {
        // Brute force implementation, could do better
        ArrayList<DictionaryCommand> allCommands = new ArrayList<>();
        for (Dictionary dict : dicts) {
            for (DictionaryCommand def : dict) {
                allCommands.add(def);
            }
        }
        return allCommands.iterator();
    }

    @Override
    public int size() {
        int result = 0;
        for (Dictionary dict : dicts) {
            result += dict.size();
        }
        return result;
    }

    @Override
    public DictionaryHelpGenerator getHelpGenerator() {
        return new CompositeCommandDictionaryHelpGenerator(dicts);
    }

    private static class CompositeCommandDictionaryHelpGenerator implements DictionaryHelpGenerator {

        /**
		 * 
		 */
		private static final long serialVersionUID = -960170507549440673L;
		private final Set<Dictionary> dictionaries;

        CompositeCommandDictionaryHelpGenerator(Set<Dictionary> dicts) {
            this.dictionaries = dicts;
        }

        @Override
        public boolean hasHelp(DictionaryCommand command) {
            for (Dictionary dic : dictionaries) {
                DictionaryHelpGenerator helpGenerator = dic.getHelpGenerator();
                if (helpGenerator != null && helpGenerator.hasHelp(command)) {
                    return true;
                }
            }
            return false;
        }

        @Override
        public String modifyHelpForCommand(DictionaryCommand command, String help, boolean compact) {
            //Can multiple DictionaryHelpGenerators contribute to a DictionaryCommand help?
            for (Dictionary dic : dictionaries) {
                DictionaryHelpGenerator helpGenerator = dic.getHelpGenerator();
                if (helpGenerator != null && helpGenerator.hasHelp(command)) {
                    help = helpGenerator.modifyHelpForCommand(command, help, compact);
                    //If we want only one contribution, we have to break here.
                    //break;
                }
            }
            return help;
        }

    }

    private static class CompositeCommandDictionaryCompleter implements DictionaryCompleter {

        /**
		 * 
		 */
		private static final long serialVersionUID = -6939187939493084396L;
		private final Set<Dictionary> dicts;

        CompositeCommandDictionaryCompleter(Set<Dictionary> dicts) {
            this.dicts = dicts;
        }

        @Override
        public int complete(String buffer, int cursor, List<CharSequence> list) {
            List<CharSequence> compositeCompletionList = new ArrayList<>();
            int returnValue = -1;
            for (Dictionary dict : dicts) {
                List<CharSequence> innerCompletionList = new ArrayList<>();
                DictionaryCompleter innerCompleter = dict.getDictionaryCompleter();
                if (innerCompleter != null) {
                    int completionValue = innerCompleter.complete(buffer, cursor, innerCompletionList);
                    if (completionValue >= returnValue && completionValue != -1) {
                        if (completionValue > returnValue) {
                            compositeCompletionList.clear();
                        }
                        compositeCompletionList.addAll(innerCompletionList);
                        returnValue = completionValue;
                    }
                }
            }
            list.addAll(compositeCompletionList);
            return returnValue;
        }
    }

}
