package org.lsst.ccs.shell;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Arrays;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import jline.console.ConsoleReader;
import jline.console.completer.Completer;
import jline.console.history.History;
import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.lsst.ccs.command.AmbiguousCommandException;
import org.lsst.ccs.command.CommandArgumentMatchException;
import org.lsst.ccs.command.CommandInvocationException;
import org.lsst.ccs.command.CommandSet;
import org.lsst.ccs.command.CommandSetBuilder;
import org.lsst.ccs.command.CompositeCommandSet;
import org.lsst.ccs.command.Dictionary;
import org.lsst.ccs.command.DictionaryCompleter;
import org.lsst.ccs.command.HelpGenerator;
import org.lsst.ccs.command.TokenizedCommand;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.utilities.pattern.PatternUtils;

/**
 * A simple shell for playing with the command parsing classes. This class is
 * designed to be run from a terminal and uses JLine to interact with the user.
 * The command shell has some built-in functionality, including the ability to
 * provide help, and the ability to do tab completion.
 *
 * @author tonyj
 */
public class JLineShell {

    private final CommandSet commands;
    private boolean exitRequested;
    private ConsoleReader reader;
    private final PrintWriter printWriter;
    private Exception lastException;
    private final Prompt prompt;
    private static final Logger logger = Logger.getLogger(JLineShell.class.getName());
    private String defaultHistoryFilePattern = "%W/shell/%A-shell.history";

    /**
     * Creates a JLineShell with the given set of user commands.
     *
     * @param userCommands The user defined commands which will be merged with
     * the built-in commands provided by the shell itself. The CommandSet passed
     * in can change dynamically (for example if it is in fact a
     * CompositeCommandSet commands can be added and removed dynamically).
     * @throws IOException If something goes horribly wrong.
     */
    public JLineShell(CommandSet userCommands) throws IOException {
        this(userCommands, new ConsoleReader(), null);
    }

    public JLineShell(CommandSet userCommands, ConsoleReader reader) {
        this(userCommands, reader, null);
    }

    public JLineShell(CommandSet userCommands, String prompt) throws IOException {
        this(userCommands, new ConsoleReader(), prompt);
    }

    public JLineShell(CommandSet userCommands, ConsoleReader reader, String prompt) {
        this.reader = reader;
        this.prompt = new Prompt(prompt);
        printWriter = new PrintWriter(reader.getOutput(), true);
        printWriter.println("Type help for list of available commands");
        CompositeCommandSet allCommands = new CompositeCommandSet();
        CommandSetBuilder builder = new CommandSetBuilder();
        allCommands.add(builder.buildCommandSet(new BuiltIns()));
        allCommands.add(userCommands);

        commands = allCommands;
        Dictionary commandDictionary = commands.getCommandDictionary();

        final DictionaryCompleter dictionaryCompleter = commandDictionary.getDictionaryCompleter();

        HelpGenerator helpGenerator = new HelpGenerator(printWriter, commandDictionary);
        allCommands.add(helpGenerator.getCommandSet());

        //Add a custom argument compelter for any commands that have such completer
        //Custom completer for "help"
        Completer completer = new Completer() {
            @Override
            public int complete(String string, int i, List<CharSequence> list) {
                return dictionaryCompleter.complete(string, i, list);
            }
        };
        reader.addCompleter(completer);
        reader.setCompletionHandler(new CommandCompletionHandler());
    }

    private void setDefaultHistoryFilePatter(String pattern) {
        this.defaultHistoryFilePattern = pattern;
    }
    
    private String getHistoryFilePattern() {
        return org.lsst.ccs.bootstrap.BootstrapResourceUtils.getBootstrapSystemProperties().getProperty("org.lsst.ccs.command.history.file", defaultHistoryFilePattern);        
    }
    /**
     * Run the command shell. This method does not return until the user exits
     * from the shell.
     *
     * @throws IOException If something goes horribly wrong.
     */
    public void run() throws IOException {
        loadHistory();
        while (!exitRequested) {
            if (prompt.isDynamic()) {
                prompt.setDynamicPrompt(commands);
            }
            String command = reader.readLine();
            if (command == null) {
                printWriter.println();
                break;
            }
            try {
                TokenizedCommand tc = new TokenizedCommand(command);
                if (!tc.isEmpty()) {
                    Object result = commands.invoke(tc);
                    if (result != null) {
                        if ( result.getClass().isArray() ) {
                            printObjectAsArray(printWriter,result);
                        } else {
                            printWriter.println(result.toString());
                        }
                    }
                }
            } catch (CommandInvocationException | CommandArgumentMatchException | AmbiguousCommandException ex  ) {
                printWriter.println("Error (type st for stacktrace): " + ex.getMessage());
                lastException = ex;
            }
        }
        saveHistory();
    }
    
    private void printObjectAsArray(PrintWriter printWriter, Object result) {
        if ( result instanceof int[] ) {
            printWriter.println(Arrays.toString((int[])result));
        } else if ( result instanceof float[] ) {
            printWriter.println(Arrays.toString((float[])result));
        } else if ( result instanceof double[] ) {
            printWriter.println(Arrays.toString((double[])result));
        } else if ( result instanceof long[] ) {
            printWriter.println(Arrays.toString((long[])result));
        } else if ( result instanceof boolean[] ) {
            printWriter.println(Arrays.toString((boolean[])result));
        } else if ( result instanceof byte[] ) {
            printWriter.println(Arrays.toString((byte[])result));
        } else if ( result instanceof short[] ) {
            printWriter.println(Arrays.toString((short[])result));
        } else if ( result instanceof char[] ) {
            printWriter.println(Arrays.toString((char[])result));
        } else {
            printWriter.println(Arrays.deepToString((Object[])result));            
        }
    }

    private void loadHistory() {
        File f = getHistoryFile();
        History history = reader.getHistory();
        if (f.canRead()) {
            try (BufferedReader in = new BufferedReader(new FileReader(f))) {
                for (;;) {
                    String line = in.readLine();
                    if (line == null) {
                        break;
                    }
                    history.add(line);
                }
            } catch (IOException x) {
                logger.log(Level.WARNING, "Unable to load command history", x);
            }
        }
    }

    private void saveHistory() {
        File f = getHistoryFile();
        try (PrintWriter out = new PrintWriter(new FileWriter(f))) {
            for (History.Entry he : reader.getHistory()) {
                out.println(he.value());
            }
        } catch (IOException x) {
            logger.log(Level.WARNING, "Unable to save command history", x);
        }
    }

    private File getHistoryFile() {
        String outputFile = PatternUtils.resolvePattern(getHistoryFilePattern());            
        File workDir = new File(outputFile.substring(0,outputFile.lastIndexOf("/")));
        workDir.mkdirs();
        return new File(outputFile);
    }

    /**
     * An enumeration of the arguments to the set command. Note that the built
     * in tab completion understands enumerations.
     */
    public enum SetCommands {

        PROMPT
    };

    /**
     * The set of built in commands.
     */
    public class BuiltIns {

        @Command(description = "Exit from the shell")
        public void exit() {
            exitRequested = true;
        }

        @Command(description = "Show command history")
        public void history() {
            History history = reader.getHistory();
            for (int i = 0; i < history.size(); i++) {
                printWriter.printf("%3d: %s\n", i+1, history.get(i));
            }
        }

        @Command(description = "Show the full stacktrace of the most recent error", alias = "st")
        public void stacktrace() {
            if (lastException != null) {
                lastException.printStackTrace(printWriter);
            }
        }

        @Command(description = "Set the value of the prompt")
        public void set(@Argument(name = "item") SetCommands what, @Argument(name = "value", defaultValue = "") String value) {
            switch (what) {
                case PROMPT:
                    prompt.setPrompt(value);
            }
        }
    }

    private class Prompt {

        private String prompt;
        private boolean isDynamic;
        private final Pattern pattern = Pattern.compile("\\$\\{(.+)\\}");

        Prompt(String prompt) {
            setPrompt(prompt);
        }

        final void setPrompt(String newPrompt) {
            this.prompt = newPrompt == null || newPrompt.isEmpty() ? ">>>" : newPrompt;
            isDynamic = pattern.matcher(prompt).find();
            reader.setPrompt(this.prompt);
        }

        void setDynamicPrompt(CommandSet cs) {
            StringBuilder newPrompt = new StringBuilder(prompt);
            Matcher matcher = pattern.matcher(newPrompt);
            while (matcher.find()) {
                String item = matcher.group(1);
                Object result;
                try {
                    result = cs.invoke(new TokenizedCommand("get " + item));
                } catch (CommandInvocationException | CommandArgumentMatchException x) {
                    result = null;
                }
                newPrompt.replace(matcher.start(), matcher.end(), result == null ? "" : result.toString());
                matcher.reset();
            }
            reader.setPrompt(newPrompt.toString());
        }

        public boolean isDynamic() {
            return isDynamic;
        }
    }

    /**
     * Temporary method waiting for feedback. This main will start the shell by
     * providing a single class name of the object the dictionary should be
     * built of. This object must have an empty constructor.
     *
     * @param argv
     * @throws Exception
     */
    public static void main(String[] argv) throws Exception {

        Options shellOptions = new Options();
        shellOptions.addOption("h", "help", false, "Print the help message");

        shellOptions.addOption("dc", "dictionaryClasses", true, "The comma separated list of classes to be used to build the command dictionary.\n"
                + "The dictionary classes must have an empty constructor in order to be loaded.");

        shellOptions.addOption("p", "prompt", true, "Set the initial prompt");

        CommandLineParser parser = new BasicParser();
        CommandLine line = parser.parse(shellOptions, argv, true);

        if (line.hasOption("help")) {
            HelpFormatter formatter = new HelpFormatter();
            formatter.printHelp(80, "CommandShell", "", shellOptions, "", true);
        } else {

            CommandSetBuilder builder = new CommandSetBuilder();
            CompositeCommandSet compositeSet = new CompositeCommandSet();

            String dictionaryClasses = line.getOptionValue("dictionaryClasses");
            String dictionaryContent = "";

            if (dictionaryClasses != null) {
                String[] classTokens = dictionaryClasses.split(",");
                for (String className : classTokens) {
                    try {
                        Class<?> c = Class.forName(className);
                        Object obj = c.newInstance();
                        compositeSet.add(builder.buildCommandSet(obj));
                        dictionaryContent += "-"+className.substring(className.lastIndexOf(".")+1);                        
                    } catch (ClassNotFoundException | IllegalAccessException | InstantiationException e) {
                        logger.log(Level.WARNING, "Skipping class " + className
                                + ". It could not be loaded or created.", e);
                    }
                }
            }
            JLineShell shell = new JLineShell(compositeSet, line.getOptionValue("prompt"));
            shell.setDefaultHistoryFilePatter("%W/shell/%A"+dictionaryContent+"-shell.history");
            shell.run();
        }
        System.exit(0);
    }
}
