package org.lsst.ccs.command;

import java.lang.reflect.Array;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.utilities.conv.InputConversionEngine;
import org.lsst.ccs.utilities.conv.TypeConversionException;

/**
 * A command with pre-parsed arguments (Objects).
 *
 * @author tonyj
 */
public class RawCommand implements BasicCommand {

    private final String commandName;
    private final List<Object> arguments;
    private static final long serialVersionUID = 8273895275408621472L;

    /**
     * Builds a Raw Command.
     *
     * @param commandName The command name.
     * @param arguments The arguments for the command
     */
    public RawCommand(String commandName, List<Object> arguments) {
        this.commandName = commandName;
        this.arguments = arguments;
    }

    /**
     * Get the root command name (the zeroth token)
     *
     * @return The command name
     */
    @Override
    public String getCommand() {
        return commandName;
    }

    /**
     * Get a specific command argument
     *
     * @param index The index of the argument
     * @return The command argument at the given index
     */
    @Override
    public Object getArgument(int index) {
        return arguments.get(index);
    }

    @Override
    public Object[] getArguments() {
        return arguments.toArray();
    }

    /**
     * Get the number of arguments associated with this command
     *
     * @return The argument count
     */
    @Override
    public int getArgumentCount() {
        return arguments.size();
    }

    /**
     * Convert a BasicCommand to a RawCommand. If the BasicCommand is already a
     * raw command it is returned unchanged. If it is a tokenized command it
     * will be converted to a RawCommand.
     *
     * @param command The input command
     * @param method The method thar will be invoked, used to find the type of the command arguments
     * @param engine The input conversion engine that will be used to convert strings to objects
     * @return The converted command
     * @throws org.lsst.ccs.command.CommandInvocationException If the command cannot be successfully converted
     */
    static RawCommand toRawCommand(BasicCommand command, Method method, InputConversionEngine engine) throws CommandInvocationException {
        if (command instanceof RawCommand) {
            return (RawCommand) command;
        } else if (command instanceof TokenizedCommand) {
            try {
                return convertToRaw((TokenizedCommand) command, method, engine);
            } catch (TypeConversionException ex) {
                throw new CommandInvocationException(ex.getMessage(), ex);
            }
        } else {
            throw new CommandInvocationException("Error: Unknown type of command " + command.getClass().getName());
        }
    }

    private static RawCommand convertToRaw(TokenizedCommand tokenizedCommand, Method method, InputConversionEngine engine) throws TypeConversionException {

        DictionaryCommand methodCommand = CommandDictionaryBuilder.getDictionaryCommandFromMethod(method);
        
        Type[] parameterTypes = method.getGenericParameterTypes();
        List<Object> args = new ArrayList(parameterTypes.length);
        boolean varArgs = method.isVarArgs();

        //The number of arguments in the tokenizedCommand can be:
        //  -A- The same as the arguments in the method
        //  -B- Less than the arguments in the method, in which case we have to use default values
        //  -C- The method has var args , in which case the number of arguments is the same or more than the arguments in the method

        // CASE C
        if (varArgs) {

            for (int i = 0; i < parameterTypes.length - 1; i++) {
                args.add(engine.convertArgToType(tokenizedCommand.getArgument(i), parameterTypes[i]));
            }
            Type varClass = parameterTypes[parameterTypes.length - 1];            
            Class elemClass = ((Class)varClass).getComponentType();
            Object theArray = Array.newInstance(elemClass, tokenizedCommand.getArgumentCount() - parameterTypes.length + 1);
            for (int j = 0; j < Array.getLength(theArray); j++) {
                Array.set(theArray, j, engine.convertArgToType(tokenizedCommand.getArgument(parameterTypes.length - 1 + j), elemClass));
            }
            args.add(theArray);
        } else {
            
            int maxArgs = Math.max(parameterTypes.length, tokenizedCommand.getArgumentCount());
            int minArgs = Math.min(parameterTypes.length, tokenizedCommand.getArgumentCount());
            //CASE A and the beginning of B
            for (int i = 0; i < minArgs; i++) {
                args.add(engine.convertArgToType(tokenizedCommand.getArgument(i), parameterTypes[i]));
            }

            // CASE B
            if (tokenizedCommand.getArgumentCount() < parameterTypes.length) {
                for (int i = minArgs; i < maxArgs; i++) {
                    args.add(engine.convertArgToType(methodCommand.getArguments()[i].getDefaultValue(), parameterTypes[i]));
                }
            }
        }
        return new RawCommand(tokenizedCommand.getCommand(), args);
    }

    @Override
    public String toString() {
        return prettyToString();
    }
    
}
