package org.lsst.ccs.command;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import static org.lsst.ccs.command.annotations.Command.CommandCategory.SYSTEM;

/**
 * A set of utilities for dealing with command dictionaries.
 *
 * @author turri
 */
public abstract class DictionaryUtils {

    static String basicHelpForDictionary(Dictionary dict) {
        return basicHelpForDictionary(dict, "");
    }

    static String basicHelpForDictionary(Dictionary dict, String indent) {
        StringBuilder helpOut = new StringBuilder();
        List<DictionaryCommand> sorted = new ArrayList<>();
        for (DictionaryCommand def : dict) {
            if (def.getCategory() != SYSTEM) {
                sorted.add(def);
            }
        }
        Collections.sort(sorted, new CommandDefinitionComparator());
        DictionaryHelpGenerator helpGenerator = dict.getHelpGenerator();

        for (DictionaryCommand def : sorted) {
            String help = DictionaryUtils.basicHelpForCommand(def);

            if (helpGenerator != null && helpGenerator.hasHelp(def)) {
                help = helpGenerator.modifyHelpForCommand(def, help, true);
            }
            helpOut.append(indent).append(help);
        }
        return helpOut.toString();
    }

    static String basicHelpForCommand(DictionaryCommand def) {
        StringBuilder builder = new StringBuilder();
        builder.append(def.getCommandName());
        for (DictionaryArgument param : def.getArguments()) {
            List<String> allowedValues = param.getAllowedValues();
            builder.append(' ').append(allowedValues.size() == 1 ? allowedValues.get(0) : param.getName());
        }
        if (def.isVarArgs()) {
            builder.append("...");
        }

        if (builder.length() <= 30) {
            for (int i = builder.length(); i < 31; i++) {
                builder.append(" ");
            }
        } else {
            builder.append("\n                               ");
        }
        builder.append(def.getDescription()).append("\n");

        if (def.getAliases().length > 0) {
            builder.append("    aliases:");
            for (String alias : def.getAliases()) {
                builder.append(" ").append(alias);
            }
            builder.append("\n");
        }
        return builder.toString();
    }

    static DictionaryCommand findCommand(Dictionary dict, String command, int argCount) {
        try {
            return dict.findCommand(new BasicCommandNullArgs(command, argCount));
        } catch (CommandArgumentMatchException ex) {
            return null;
        }
    }

    static boolean containsCommand(Dictionary dict, String command, int argCount) {
        try {
            return dict.containsCommand(new BasicCommandNullArgs(command, argCount));
        } catch (CommandArgumentMatchException ex) {
            return false;
        }
    }

    static boolean containsDictionaryCommand(Dictionary dict, DictionaryCommand newCmd) {
        
        for ( DictionaryCommand oldCommand : dict ) {
            ArrayList<String> cmdNameOrAliases = new ArrayList<>();
            cmdNameOrAliases.add(oldCommand.getCommandName());
            String[] aliases = oldCommand.getAliases();
            if (aliases != null) {
                cmdNameOrAliases.addAll(Arrays.asList(aliases));
            }
            
            for (String cmdNameOrAlias : cmdNameOrAliases) {
                //Check if there is any match with either the command name or an alias
                if (newCmd.getCommandName().equals(cmdNameOrAlias)) {
                    
                    if ( newCmd.isVarArgs() || oldCommand.isVarArgs() ) {
                        return true;
                    }
                                        
                    if ( newCmd.getArguments().length == oldCommand.getArguments().length ) {
                        boolean isOk = false;
                        //If the commands have the same name and number of arguments,
                        //they can still be different if the arguments have disjoined
                        //sets of allowed values for their arguments
                        int nArgs = newCmd.getArguments().length;
                        for ( int i = 0; i < nArgs; i++ ) {
                            DictionaryArgument newArg = newCmd.getArguments()[i];
                            DictionaryArgument oldArg = oldCommand.getArguments()[i];
                            if ( newArg.getAllowedValues().isEmpty() || oldArg.getAllowedValues().isEmpty() ) {
                                return true;
                            }
                            List<String> allowedValues = new ArrayList<>(newArg.getAllowedValues());
                            allowedValues.retainAll( oldArg.getAllowedValues() );
                            if ( ! allowedValues.isEmpty() ) {
                                return true;
                            } else {
                                isOk = true;
                                break;
                            }
                        }
                        if ( isOk ) {
                            continue;
                        }
                        return true;
                    }            
                }
            }
        }
        return false;        
    }
    
    
    
    /**
     * Test if the given command dictionary entry matches the given command.
     * This method takes into account the presence of default arguments, varargs,
     * and commands which take a specific enumerated set of legal values.
     *
     * @param def The dictionary command entry
     * @param tc The command to test
     * @return <code>true</code> if the command matches the dictionary command entry
     */
    static boolean commandMatch(DictionaryCommand def, BasicCommand tc) throws CommandArgumentMatchException {

        String command = tc.getCommand();
        int argumentCount = tc.getArgumentCount();

        ArrayList<String> cmdNameOrAliases = new ArrayList<>();
        cmdNameOrAliases.add(def.getCommandName());
        String[] aliases = def.getAliases();
        if (aliases != null) {
            cmdNameOrAliases.addAll(Arrays.asList(aliases));
        }
        for (String cmdNameOrAlias : cmdNameOrAliases) {
            //Check if there is any match with either the command name or an alias
            if (cmdNameOrAlias.equals(command)) {
                //Arguments matching:
                //   -> Same number of arguments
                //   -> More arguments with the extra ones with default values
                //   -> Dictionary command less arguments but has var args
                boolean argumentCountOk = false;
                int cmdNumberOfArguments = def.getArguments().length;
                if (argumentCount == cmdNumberOfArguments) {
                    argumentCountOk = true;
                } else if (argumentCount >= cmdNumberOfArguments - 1 && def.isVarArgs()) {
                    argumentCountOk = true;
                } else if (argumentCount < cmdNumberOfArguments) {
                    argumentCountOk = true;
                    for (int i = argumentCount; i < cmdNumberOfArguments; i++) {
                        if (def.getArguments()[i].getDefaultValue() == null) {
                            argumentCountOk = false;
                            break;
                        }
                    }
                }
                //If there is a match with the number of arguments
                //check if there is a match with the argument's value (if provided)
                //with the corresponding arguments allowed values.
                //If the match fails, false is returned;
                if (argumentCountOk) {
                    if (argumentCount == 0) {
                        return true;
                    }
                    Object[] argValues = tc.getArguments();
                    if (argValues != null) {
                        boolean argMatch = true;
                        for (int i = 0; i < argValues.length; i++) {
                            if (i >= def.getArguments().length) {
                                return true;
                            }
                            List<String> allowedValues = def.getArguments()[i].getAllowedValues();
                            if (allowedValues != null && !allowedValues.isEmpty()) {
                                argMatch = false;
                                for (String allowedValue : allowedValues) {
                                    if (allowedValue.toLowerCase().equals(argValues[i].toString().toLowerCase())) {
                                        argMatch = true;
                                        break;
                                    }
                                }
                            }
                            if (!argMatch) {
                                throw new CommandArgumentMatchException("Illegal value \""+argValues[i]+"\" for argument \""+def.getArguments()[i].getName()+"\"."+
                                        "Allowed values are: "+allowedValues);
                            }
                        }
                        return true;
                    } else {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    private static class BasicCommandNullArgs implements BasicCommand {

        private final String commandName;
        private final int argumentCount;

        BasicCommandNullArgs(String commandName, int argumentCount) {
            this.commandName = commandName;
            this.argumentCount = argumentCount;
        }

        @Override
        public int getArgumentCount() {
            return argumentCount;
        }

        @Override
        public String getCommand() {
            return commandName;
        }

        @Override
        public Object getArgument(int i) {
            return null;
        }

        @Override
        public Object[] getArguments() {
            return null;
        }

        @Override
        public String toString() {
            return prettyToString();
        }

    }

    /**
     * A comparator used for putting commands into alphabetical order
     */
    private static class CommandDefinitionComparator implements Comparator<DictionaryCommand> {

        @Override
        public int compare(DictionaryCommand o1, DictionaryCommand o2) {
            return o1.getCommandName().compareTo(o2.getCommandName());
        }
    }
}
