package org.lsst.ccs.command;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

/**
 * Takes a single object and builds a command set from its annotated methods.
 * The command set consists of two parts, the CommandDictionary, which is
 * serializable and the command invoker, which contains references to the object
 * and methods and is not serializable.
 *
 * @author tonyj
 */
public class CommandSetBuilder {

    private final InputConversionEngine engine = new InputConversionEngine();

    /**
     * Build a command set from an objects annotations.
     *
     * @param object The object from which annotations will be extracted, and to
     * which invocation requests will be forwarded.
     * @return A CommandSet representing the commands found in the object.
     */
    public CommandSet buildCommandSet(Object object) {
        Class targetClass = object.getClass();
        CommandDictionaryBuilder dictionary = new CommandDictionaryBuilder(targetClass);
        CommandSetImplementation commandSet = new CommandSetImplementation(dictionary, object);
        return commandSet;
    }

    private class CommandSetImplementation implements CommandSet {

        private final CommandDictionaryBuilder dict;
        private final Object target;

        private CommandSetImplementation(CommandDictionaryBuilder dict, Object target) {
            this.dict = dict;
            this.target = target;
        }

        @Override
        public Dictionary getCommandDictionary() {
            return dict.getCommandDictionary();
        }

        @Override
        public Object invoke(BasicCommand command) throws CommandInvocationException, CommandArgumentMatchException {
            Method method = dict.getMethod(command);
            if (method == null) {
                throw new CommandInvocationException("Error: No handler found for command %s with %d arguments", command.getCommand(), command.getArgumentCount());
            }
            return invoke(target, method, command);
        }

        private Object invoke(Object target, Method method, BasicCommand command) throws CommandInvocationException {

            try {
                RawCommand raw = RawCommand.toRawCommand(command, method, engine);
                return method.invoke(target, raw.getArguments());
            } catch (IllegalAccessException | IllegalArgumentException ex) {
                throw new CommandInvocationException("Error: Can't invoke command", ex);
            } catch (InvocationTargetException ex) {
                throw new CommandInvocationException(ex);
            }
        }
    }
}
