package org.lsst.ccs.command;

import java.util.List;
import org.lsst.ccs.command.StringTokenizer.Token;

/**
 * A command line that has been split into tokens. This could in future be
 * expanded to also support command options (beginning with - or --)
 *
 * @author tonyj
 */
public class TokenizedCommand implements BasicCommand {

    private final List<Token> tokens;
    private final String[] args;
    private static final long serialVersionUID = -687119049448004978L;

    /**
     * Builds a Tokenized command from an unparsed string.
     *
     * @param command The command string to be tokenized.
     */
    public TokenizedCommand(String command) {
        tokens = StringTokenizer.tokenize(command);
        int argCount = getArgumentCount();
        args = new String[argCount];
        for ( int i = 0; i < argCount; i++) {
            args[i] = getArgument(i);
        }
    }

    /**
     * Get the root command name (the zeroth token)
     *
     * @return The command name
     */
    @Override
    public String getCommand() {
        return tokens.get(0).getString();
    }

    /**
     * Get the start position on the line of the command name.
     *
     * @return The position of the command
     */
    int getCommandLocation() {
        return tokens.get(0).getLocation();
    }

    /**
     * Get a specific command argument
     *
     * @param index The index of the argument
     * @return The command argument at the given index
     */
    @Override
    public String getArgument(int index) {
        return tokens.get(index + 1).getString();
    }

    @Override
    public String[] getArguments() {
        return args;
    }
    
    /**
     * The start point in the original command line of a specific argument.
     *
     * @param index The index of the argument
     * @return The position of the argument
     */
    int getArgumentLocation(int index) {
        return tokens.get(index + 1).getLocation();
    }

    /**
     * Get the number of arguments associated with this command
     *
     * @return The argument count
     */
    @Override
    public int getArgumentCount() {
        return Math.max(tokens.size() - 1, 0);
    }

    /**
     * Test if the command is completely empty (no tokens)
     *
     * @return <code>true</code> only if the command is empty.
     */
    public boolean isEmpty() {
        return tokens.isEmpty();
    }
    
    @Override
    public String toString() {
        return "TokenizedCommand: "+getCommand()+" with "+getArgumentCount()+" arguments.";
    }
}
