package org.lsst.ccs.command;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import org.lsst.ccs.command.annotations.Command;

/**
 * A CompositeCommandSet that provides a route to activate another CommandSet.
 * When the RoutingCommandSet is active, it adds to its own command set
 * the route hidden CommandSet, making it available for straight execution.
 *
 * @author turri
 */
public class RoutingCommandSet implements CommandSet {

    private final RoutingDictionary dict;
    private final String route;
    private final CommandSet cs;

    public RoutingCommandSet(String route, CommandSet cs) {
        this.route = route;
        this.cs = cs;
        dict = new RoutingDictionary(route, cs.getCommandDictionary());
    }

    String getRoute() {
        return route;
    }

    CommandSet getRouteCommandSet() {
        return cs;
    }

    @Override
    public Dictionary getCommandDictionary() {
        return dict;
    }

    @Override
    public Object invoke(BasicCommand command) throws CommandInvocationException {
        DictionaryCommand dc = dict.findCommand(command);
        if (dc == null) {
            throw new CommandInvocationException("Error: Cound not invoke command " + command.getCommand() + " on target " + getRoute());
        }

        if (command.getArgumentCount() == 0) {
            throw new CommandInvocationException("Error: To change target type the command \"set target TARGET_NAME\"");
        }

        String route = command.getCommand();
        TokenizedCommand tc = (TokenizedCommand) command;
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < tc.getArgumentCount(); i++) {
            sb.append(tc.getArgument(i));
            sb.append(" ");
        }
        TokenizedCommand routedCommand = new TokenizedCommand(sb.toString());
        return getRouteCommandSet().invoke(routedCommand);

    }

    private static class RoutingDictionary implements Dictionary {

        private final LinkedHashMap<String, DictionaryCommand> routeMap = new LinkedHashMap<>();
        private final RoutingDictionaryHelpGenerator helpGenerator = new RoutingDictionaryHelpGenerator(routeMap);
        private final Dictionary innerDictionary;
        private final RoutingDictionaryCompleter completer;
        private final String route;

        RoutingDictionary(String route, Dictionary dict) {
            routeMap.put(route, new RoutingCommand(route, dict));
            innerDictionary = dict;
            this.route = route;
            completer = new RoutingDictionaryCompleter(this);
        }

        RoutingCommand getRouteCommand(String route) {
            return (RoutingCommand) routeMap.get(route);
        }

        Dictionary getInnerDictionary() {
            return innerDictionary;
        }

        String getRoute() {
            return route;
        }

        @Override
        public boolean containsCommand(BasicCommand tc) {
            return routeMap.containsKey(tc.getCommand());
        }

        @Override
        public DictionaryCommand findCommand(BasicCommand tc) {
            return routeMap.get(tc.getCommand());
        }

        @Override
        public int size() {
            return routeMap.size();
        }

        @Override
        public Iterator<DictionaryCommand> iterator() {
            return routeMap.values().iterator();
        }

        @Override
        public DictionaryHelpGenerator getHelpGenerator() {
            return helpGenerator;
        }

        @Override
        public DictionaryCompleter getDictionaryCompleter() {
            return completer;
        }

    }

    private static class RoutingDictionaryCompleter implements DictionaryCompleter {

        private final DictionaryCompleter routingDictionaryCompleter;
        private final DictionaryCompleter innerDictionaryCompleter;
        private final String route;

        RoutingDictionaryCompleter(RoutingDictionary dict) {
            routingDictionaryCompleter = new DefaultDictionaryCompleter(dict);
            innerDictionaryCompleter = dict.getInnerDictionary().getDictionaryCompleter();
            route = dict.getRoute();
        }

        @Override
        public int complete(String buffer, int index, List<CharSequence> list) {
            TokenizedCommand tc = new TokenizedCommand(buffer.substring(0, index));
            int argCount = tc.getArgumentCount();
            boolean endsWithWhiteSpace = !tc.isEmpty() && Character.isWhitespace(buffer.charAt(index - 1));

            if (endsWithWhiteSpace) {
                argCount++;
            }

            if (argCount > 1 && buffer.startsWith(route)) {
                if (innerDictionaryCompleter != null) {
                    int extraIndex = route.length() + 1;
                    String newBuffer = buffer.substring(extraIndex);
                    int newIndex = index - extraIndex;
                    int completionValue = innerDictionaryCompleter.complete(newBuffer, newIndex, list);
                    return completionValue == -1 ? completionValue : completionValue + route.length() + 1;
                }
            } else {
                int ret = routingDictionaryCompleter.complete(buffer, index, list);
                return ret;
            }
            return index;
        }

    }

    private static class RoutingDictionaryHelpGenerator implements DictionaryHelpGenerator {

        private final HashMap<String, DictionaryCommand> routeMap;

        public RoutingDictionaryHelpGenerator(HashMap<String, DictionaryCommand> routeMap) {
            this.routeMap = routeMap;
        }

        @Override
        public boolean hasHelp(DictionaryCommand command) {
            return command instanceof RoutingCommand && routeMap.containsValue(command);
        }

        @Override
        public String modifyHelpForCommand(DictionaryCommand command, String help, boolean compact) {
            help = "(t) " + help;
            if (!compact) {
                help += "Available commands: \n";
                help += DictionaryUtils.basicHelpForDictionary(((RoutingCommand) command).getDictionary(), "     ");
            }
            return help;
        }
    }

    private static class RoutingCommand implements DictionaryCommand {

        private final String route;
        private final Dictionary dict;
        private final DictionaryArgument[] routingArgument;

        RoutingCommand(String route, Dictionary dict) {
            this.route = route;
            this.dict = dict;
            routingArgument = new DictionaryArgument[]{new RoutingCommandArgument(dict), new RoutingCommandArgumentsArgument()};
        }

        Dictionary getDictionary() {
            return dict;
        }

        @Override
        public DictionaryArgument[] getArguments() {
            return routingArgument;
        }

        @Override
        public String getCommandName() {
            return route;
        }

        @Override
        public String getDescription() {
            return "Sends commands to target's (" + route + ") dictionary";
        }

        @Override
        public Command.CommandType getType() {
            return Command.CommandType.QUERY;
        }

        @Override
        public boolean isVarArgs() {
            return true;
        }

        @Override
        public int getLevel() {
            return Command.NORMAL;
        }

        @Override
        public Command.CommandCategory getCategory() {
            return Command.CommandCategory.USER;
        }
        
        private static class RoutingCommandArgumentsArgument implements DictionaryArgument {

            @Override
            public String getDescription() {
                return "The arguments for the command";
            }

            @Override
            public String getName() {
                return "arguments...";
            }

            @Override
            public String getSimpleType() {
                return "String";
            }

            @Override
            public String getType() {
                return "java.lang.String";
            }

            @Override
            public List<String> getAllowedValues() {
                return Collections.EMPTY_LIST;
            }

            @Override
            public String getDefaultValue() {
                return null;
            }
        }

        private static class RoutingCommandArgument implements DictionaryArgument {

            private final Dictionary dict;

            RoutingCommandArgument(Dictionary dict) {
                this.dict = dict;
            }

            @Override
            public String getDescription() {
                return "The command to execute";
            }

            @Override
            public String getName() {
                return "command";
            }

            @Override
            public String getSimpleType() {
                return "String";
            }

            @Override
            public String getType() {
                return "java.lang.String";
            }

            @Override
            public List<String> getAllowedValues() {
                ArrayList<String> allowedValues = new ArrayList<>();
                for (DictionaryCommand command : dict) {
                    allowedValues.add(command.getCommandName());
                }
                return allowedValues;
            }

            @Override
            public String getDefaultValue() {
                return null;
            }
        }
    }
}
