package org.lsst.ccs.shell;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.List;
import jline.console.ConsoleReader;
import jline.console.completer.Completer;
import jline.console.history.History;
import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.lsst.ccs.command.CommandInvocationException;
import org.lsst.ccs.command.CommandSet;
import org.lsst.ccs.command.CommandSetBuilder;
import org.lsst.ccs.command.CompositeCommandSet;
import org.lsst.ccs.command.Dictionary;
import org.lsst.ccs.command.DictionaryCompleter;
import org.lsst.ccs.command.HelpGenerator;
import org.lsst.ccs.command.TokenizedCommand;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;

/**
 * A simple shell for playing with the command parsing classes. This class is
 * designed to be run from a terminal and uses JLine to interact with the user.
 * The command shell has some built-in functionality, including the ability to
 * provide help, and the ability to do tab completion.
 *
 * @author tonyj
 */
public class JLineShell {

    private final CommandSet commands;
    private boolean exitRequested;
    private final ConsoleReader reader;
    private final PrintWriter printWriter;
    private CommandInvocationException lastException;

    /**
     * Creates a JLineShell with the given set of user commands.
     *
     * @param userCommands The user defined commands which will be merged with
     * the built-in commands provided by the shell itself. The CommandSet passed
     * in can change dynamically (for example if it is in fact a
     * CompositeCommandSet commands can be added and removed dynamically).
     * @throws IOException If something goes horribly wrong.
     */
    public JLineShell(CommandSet userCommands) throws IOException {
        this(userCommands, new ConsoleReader(), null);
    }

    public JLineShell(CommandSet userCommands, ConsoleReader reader) {
        this(userCommands, reader, null);
    }

    public JLineShell(CommandSet userCommands, String prompt) throws IOException {
        this(userCommands, new ConsoleReader(), prompt);
    }

    public JLineShell(CommandSet userCommands, ConsoleReader reader, String prompt) {
        this.reader = reader;
        reader.setPrompt(prompt != null ? prompt : ">>>");
        printWriter = new PrintWriter(reader.getOutput(), true);
        printWriter.println("Type help for list of available commands");
        CompositeCommandSet allCommands = new CompositeCommandSet();
        CommandSetBuilder builder = new CommandSetBuilder();
        allCommands.add(builder.buildCommandSet(new BuiltIns()));
        allCommands.add(userCommands);
        Dictionary commandDictionary = allCommands.getCommandDictionary();
        final DictionaryCompleter dictionaryCompleter = new DictionaryCompleter(commandDictionary);
        allCommands.add(builder.buildCommandSet(new HelpGenerator(printWriter, commandDictionary)));
        commands = allCommands;
        Completer completer = new Completer() {
            @Override
            public int complete(String string, int i, List<CharSequence> list) {
                return dictionaryCompleter.complete(string, i, list);
            }
        };
        reader.addCompleter(completer);
        reader.setCompletionHandler(new CommandCompletionHandler());
    }

    /**
     * Run the command shell. This method does not return until the user exits
     * from the shell.
     *
     * @throws IOException If something goes horribly wrong.
     */
    public void run() throws IOException {
        while (!exitRequested) {
            String command = reader.readLine();
            if (command == null) {
                printWriter.println();
                break;
            }
            try {
                TokenizedCommand tc = new TokenizedCommand(command);
                if (!tc.isEmpty()) {
                    Object result = commands.invoke(tc);
                    if (result != null) {
                        printWriter.println(result.toString());
                    }
                }
            } catch (CommandInvocationException ex) {
                printWriter.println(ex.getMessage());
                lastException = ex;
            }
        }
    }

    /**
     * An enumeration of the arguments to the set command. Note that the built
     * in tab completion understands enumerations.
     */
    public enum SetCommands {

        PROMPT
    };

    /**
     * The set of built in commands.
     */
    public class BuiltIns {

        @Command(description = "Exit from the shell")
        public void exit() {
            exitRequested = true;
        }

        @Command(description = "Show command history")
        public void history() {
            History history = reader.getHistory();
            for (int i = 0; i < history.size(); i++) {
                printWriter.printf("%3d: %s\n", i, history.get(i));
            }
        }

        @Command(description = "Show the full stacktrace of the most recent error", alias = "st")
        public void stacktrace() {
            if (lastException != null) {
                lastException.printStackTrace(printWriter);
            }
        }

        @Command(description = "Modify various settings")
        public void set(@Argument(name = "item") SetCommands what, @Argument(name = "value") String value) {
            switch (what) {
                case PROMPT:
                    reader.setPrompt(value);
            }
        }
    }

    /**
     * Temporary method waiting for feedback. This main will start the shell by
     * providing a single class name of the object the dictionary should be
     * built of. This object must have an empty constructor.
     *
     * @param argv
     * @throws Exception
     */
    public static void main(String[] argv) throws Exception {

        Options shellOptions = new Options();
        shellOptions.addOption("h","help",false,"Print the help message");
                       
        shellOptions.addOption("dc","dictionaryClasses",true,"The comma separated list of classes to be used to build the command dictionary.\n"
                + "The dictionary classes must have an empty constructor in order to be loaded.");
        
        shellOptions.addOption("p", "prompt", true, "Set the initial prompt");

        CommandLineParser parser = new BasicParser();
        CommandLine line = parser.parse(shellOptions, argv, true);

        if (line.hasOption("help")) {
            HelpFormatter formatter = new HelpFormatter();
            formatter.printHelp(80,"CommandShell", "", shellOptions, "", true);
        } else {

            CommandSetBuilder builder = new CommandSetBuilder();
            CompositeCommandSet compositeSet = new CompositeCommandSet();

			String dictionaryClasses = line.getOptionValue("dictionaryClasses");
			if (dictionaryClasses != null) {
				String[] classTokens = dictionaryClasses.split(",");
				for (int i = 0; i < classTokens.length; i++) {
					String className = classTokens[i];
					try {
						Class<?> c = Class.forName(className);
						Object obj = c.newInstance();
						compositeSet.add(builder.buildCommandSet(obj));
					} catch (Exception e) {
						System.out.println("** Skipping class " + className
								+ ". It could not be loaded or created.");
						e.printStackTrace();
					}
				}

			}
            JLineShell shell = new JLineShell(compositeSet, line.getOptionValue("prompt"));
            shell.run();
        }
        System.exit(0);
    }
}
