package org.lsst.ccs.command;

import org.lsst.ccs.command.annotations.Command;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

/**
 * This class consists of a CommandDictionary plus a map of which methods should
 * be called as a result of each command. It does not require a reference to any
 * particular object, just a class.
 *
 * Note that since this object is not a CommandSet, the utilities for combining
 * and working with CommandSets cannot be used with this class.
 *
 * @author tonyj
 */
// change history: bamade added implements to new interface LocalCommandDictionary
public class CommandDictionaryBuilder implements LocalCommandDictionary {

	private MethodBasedCommandDictionary dict;
	private final Map<DictionaryCommand, Method> methods = new HashMap<>();

	/**
	 * Build a command dictionary for the given class.
	 *
	 * @param klass The class to build the dictionary from
	 */
	CommandDictionaryBuilder(Class klass) {
		init(klass);
	}

	/**
	 * Get the command dictionary.
	 *
	 * @return The command dictionary.
	 */
	public Dictionary getCommandDictionary() {
		return dict;
	}

	/**
	 * Find the method which should be invoked as a result of a particular
	 * command.
	 *
	 * @param command The command to be invoked
	 * @return The method to be called to invoke the command, or
	 * <code>null</code> if the command is not known.
	 */
	public Method getMethod(BasicCommand command) {
	    //todo: note by bernard .. the way I use it we are doing twice the findCommand!!!
		// so I added the method below
		DictionaryCommand dc = dict.findCommand(command);
		if (dc == null) {
			return null;
		}
		return methods.get(dc);
	}

	/***
	 * Finds a <TT>Method</TT> which is linked to a <TT>DictionaryCommand</TT>.
	 * @param dc a dictionary command that should be extracted from the current <TT>Dicionary</TT>
	 * @return The method to be called to invoke the command, or
	 * <code>null</code> if the command is not known.
	 */
	public Method getMethod(DictionaryCommand dc) {
		if (dc == null) {
			return null;
		}
		return methods.get(dc);
	}


    private static Method[] arrayModel = new Method[0] ;
    /*
    * return the list of methods that are commands
     */
    public Collection<Method> getMethods() {
         return methods.values() ;
    }

	private void init(Class targetClass) {
		dict = new MethodBasedCommandDictionary();
		for (Method targetMethod : targetClass.getMethods()) {
			Method annotatedMethod = getAnnotationWithInheritance(targetMethod, Command.class);
			if (annotatedMethod != null) {
				Command annotation = annotatedMethod.getAnnotation(Command.class);
				if (annotation != null) {
					MethodBasedDictionaryCommand dc = new MethodBasedDictionaryCommand(annotatedMethod, annotation);
					dict.add(dc);
					methods.put(dc, targetMethod);
				}
			}
		}
	}

	/**
	 * Java does not currently support inheritance of method annotations
	 * (only Class annotations). This method works around this be searching
	 * for annotations for a particular method on super classes.
	 *
	 * @param method The method on which to search
	 * @param annotationClass The annotation subclass to search for
	 * @return The annotated method if found, or <code>null</code> if not
	 * found
	 */
	private static <T extends Annotation> Method getAnnotationWithInheritance(Method method, Class<T> annotationClass) {
		for (;;) {
			T annotation = method.getAnnotation(annotationClass);
			if (annotation != null) {
				return method;
			}
			Class superClass = method.getDeclaringClass().getSuperclass();
			if (superClass == null) {
				break;
			}
			try {
				method = superClass.getMethod(method.getName(), method.getParameterTypes());
			} catch (NoSuchMethodException ex) {
				break;
			}
		}
		return null;
	}
}
