package org.lsst.ccs.command;

import java.lang.reflect.Array;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

/**
 * A command with pre-parsed arguments (Objects).
 *
 * @author tonyj
 */
public class RawCommand implements BasicCommand {

    private final String commandName;
    private final List<Object> arguments;

    /**
     * Builds a Raw Command.
     *
     * @param commandName The command name.
     * @param arguments The arguments for the command
     */
    public RawCommand(String commandName, List<Object> arguments) {
        this.commandName = commandName;
        this.arguments = arguments;
    }

    /**
     * Get the root command name (the zeroth token)
     *
     * @return The command name
     */
    @Override
    public String getCommand() {
        return commandName;
    }

    /**
     * Get a specific command argument
     *
     * @param index The index of the argument
     * @return The command argument at the given index
     */
    public Object getArgument(int index) {
        return arguments.get(index);
    }
    
    public Object[] getArguments() {
        return arguments.toArray();
    }

    /**
     * Get the number of arguments associated with this command
     *
     * @return The argument count
     */
    @Override
    public int getArgumentCount() {
        return arguments.size();
    }

    /**
     * Convert a BasicCommand to a RawCommand. If the BasicCommand is already a
     * raw command it is returned unchanged. If it is a tokenized command it
     * will be converted to a RawCommand.
     *
     * @param command The input command
     * @param method The method thar will be invoked, used to find the type of the command arguments
     * @param engine The input conversion engine that will be used to convert strings to objects
     * @return The converted command
     * @throws org.lsst.ccs.command.CommandInvocationException If the command cannot be successfully converted
     */
    public static RawCommand toRawCommand(BasicCommand command, Method method, InputConversionEngine engine) throws CommandInvocationException {
        if (command instanceof RawCommand) {
            return (RawCommand) command;
        } else if (command instanceof TokenizedCommand) {
            return convertToRaw((TokenizedCommand) command, method, engine);
        } else {
            throw new CommandInvocationException("Error: Unknown type of command " + command.getClass().getName());
        }
    }

    private static RawCommand convertToRaw(TokenizedCommand tokenizedCommand, Method method, InputConversionEngine engine) throws CommandInvocationException {
        Class<?>[] parameterTypes = method.getParameterTypes();
        List<Object> args = new ArrayList(parameterTypes.length);
        boolean varArgs = method.isVarArgs();
        for (int i = 0; i < parameterTypes.length; i++) {
            if (varArgs && i == parameterTypes.length - 1) {
                Class varClass = parameterTypes[i];
                Class elemClass = varClass.getComponentType();
                Object theArray = Array.newInstance(elemClass, tokenizedCommand.getArgumentCount() - parameterTypes.length + 1);
                for (int j = 0; j < Array.getLength(theArray); j++) {
                    Array.set(theArray, j, engine.convertArgToType(tokenizedCommand.getArgument(i + j), elemClass));
                }
                args.add(theArray);
            } else {
                args.add(engine.convertArgToType(tokenizedCommand.getArgument(i), parameterTypes[i]));
            }
        }
        return new RawCommand(tokenizedCommand.getCommand(), args);
    }
}
