package org.lsst.ccs.command;

import org.lsst.ccs.utilities.beanutils.Optional;
import org.lsst.ccs.utilities.structs.TreeBranch;

import javax.swing.tree.TreeNode;
import java.util.Iterator;
import java.util.LinkedHashMap;

/**
 * Client codes can receive a complete Tree of DictionaryContext from a Subsystem.
 * It can then build this object to read informations about available dictionaries.
 *
 * @author bamade
 * @ImplSpec : is the "top node" the node for the "main" Module or for the Subsystem itself?
 */
// Date: 08/01/2014

public class DictionariesTree implements Iterable<DictionaryContext> {
    private final TreeBranch<DictionaryContext> topNode;
    private LinkedHashMap<String, TreeBranch<DictionaryContext>> map;

    public DictionariesTree(TreeBranch<DictionaryContext> topNode) {
        this.topNode = topNode;
        Iterator<TreeBranch<DictionaryContext>> iterator = topNode.nodeIterator();
        while (iterator.hasNext()) {
            TreeBranch<DictionaryContext> node = iterator.next();
            DictionaryContext dictionaryContext = node.getContent();
            map.put(dictionaryContext.getName(), node);
        }
    }

    /**
     * gets a <TT>Dictionary</TT> linked to a Component.
     *
     * @param componentName
     * @return an <TT>Optional</TT> Dictionary (just in case the name is wrong!)
     */
    public Optional<Dictionary> getDictionaryFor(String componentName) {
        TreeBranch<DictionaryContext> node = map.get(componentName);
        if (node != null) {
            return Optional.of(node.getContent().getDictionary());
        }
        return Optional.empty();
    }

    /**
     * Since the <TT>TreeBranch</TT> class is a <TT>TreeNode</TT>
     * this method is useful for GUIs in need of a JTree.
     * or for all methods that "walk" trees.
     * <BR/>
     * The tree has the same structure as the subsystem description file.
     *
     * @return
     */
    public TreeBranch<DictionaryContext> getTopNode() {
        return topNode;
    }

    public TreeNode getNodeFor(String componentName) {
        return map.get(componentName);
    }

    /**
     * @return all the <TT>DictionaryContext</TT> in preOrder
     * (that is with the structure defined in the subsystem description file)
     */
    @Override
    public Iterator<DictionaryContext> iterator() {
        return topNode.iterator();
    }

    /**
     * @return the names of the component of the subsystem in preOrder.
     */
    public Iterator<String> namesIterator() {
        return new Iterator<String>() {
            Iterator<DictionaryContext> ctxIterator = iterator();

            @Override
            public boolean hasNext() {
                return ctxIterator.hasNext();
            }

            @Override
            public String next() {
                DictionaryContext dictCtx = ctxIterator.next();
                return dictCtx.getName();
            }

            @Override
            public void remove() {
                throw new UnsupportedOperationException("remove in nameIterator");
            }
        };
    }


}
