package org.lsst.ccs.command;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.Arrays;

/**
 * An implementation of DictionaryCommand based on a single annotated method.
 *
 * @author turri
 */
class MethodBasedDictionaryCommand implements DictionaryCommand {

    private final String description;
    private final String[] aliases;
    private final DictionaryArgument[] params;
    private final Command.CommandType type;
    private final String name;
    private final boolean hasVarArgs;

    /**
     * Create a command definition from a method and associated annotation.
     *
     * @param method     The method providing the command implementation
     * @param annotation The annotation on the method
     */
    MethodBasedDictionaryCommand(Method method, Command annotation) {
        this.description = annotation.description();
        this.aliases = splitAliases(annotation.alias());
        this.type = annotation.type();
        this.name = annotation.name().isEmpty() ? method.getName() : annotation.name();
        this.hasVarArgs = method.isVarArgs();

        Class[] types = method.getParameterTypes();
        Annotation[][] parAnnotations = method.getParameterAnnotations();

        params = new MethodBasedDictionaryArgument[types.length];

        for (int i = 0; i < types.length; i++) {

            String parName = "arg" + i;
            String parDescription = "";
            for (Annotation a : parAnnotations[i]) {
                if (a instanceof Argument) {
                    Argument paramAnnotation = (Argument) a;
                    parName = paramAnnotation.name().isEmpty() ? parName : paramAnnotation.name();
                    parDescription = paramAnnotation.description();
                    break;
                }
            }
            Class parameterType = hasVarArgs && i == types.length - 1 ? types[i].getComponentType() : types[i];
            params[i] = new MethodBasedDictionaryArgument(parName, parameterType, parDescription);
        }


    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public String[] getAliases() {
        return aliases;
    }

    @Override
    public DictionaryArgument[] getArguments() {
        return params;
    }

    @Override
    public Command.CommandType getType() {
        return type;
    }

    @Override
    public String getCommandName() {
        return name;
    }

    @Override
    public boolean isVarArgs() {
        return hasVarArgs;
    }

    /**
     * Test if value matches this command (either the command itself or one if
     * its aliases).
     *
     * @param value The value to test
     * @return <code>true</code> if the value matches
     */
    @Override
    public boolean matchesCommandOrAlias(String value) {
        if (name.equals(value)) {
            return true;
        }
        for (String alias : aliases) {
            if (alias.equals(value)) {
                return true;
            }
        }
        return false;
    }

    private String[] splitAliases(String alias) {
        return alias.length() > 0 ? alias.split("\\s?,\\s?") : NO_ALIASES;
    }

    @Override
    public String toString() {
        return "MethodBasedDictionaryCommand{" +
                "description='" + description + '\'' +
                ", aliases=" + Arrays.toString(aliases) +
                ", params=" + Arrays.toString(params) +
                ", type=" + type +
                ", name='" + name + '\'' +
                ", hasVarArgs=" + hasVarArgs +
                '}';
    }
}
