package org.lsst.ccs.command;

import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.command.annotations.Command;

/**
 * Takes a single object and builds a command set from its annotated methods.
 * The command set consists of two parts, the CommandDictionary, which is
 * serializable and the command invoker, which contains references to the object
 * and methods and is not serializable.
 *
 * @author tonyj
 */
public class CommandSetBuilder {

    private InputConversionEngine engine = new InputConversionEngine();

    /**
     * Build a command set from an objects annotations.
     *
     * @param object The object from which annotations will be extracted, and to
     * which invocation requests will be forwarded.
     * @return A CommandSet representing the commands found in the object.
     */
    public CommandSet buildCommandSet(Object object) {
        Class targetClass = object.getClass();
        MethodBasedCommandDictionary dict = new MethodBasedCommandDictionary();
        CommandSetImplementation commandSet = new CommandSetImplementation(dict, object);
        for (Method method : targetClass.getMethods()) {
            Command annotation = method.getAnnotation(Command.class);
            if (annotation != null) {
                MethodBasedDictionaryCommand dc = new MethodBasedDictionaryCommand(method, annotation);
                dict.add(dc);
                commandSet.add(dc,method);
            }
        }
        return commandSet;
    }

    private class CommandSetImplementation implements CommandSet {

        private final Dictionary dict;
        private final Object target;
        private final Map<DictionaryCommand,Method> methods = new HashMap<>();

        private CommandSetImplementation(Dictionary dict, Object target) {
            this.dict = dict;
            this.target = target;
        }

        @Override
        public Dictionary getCommandDictionary() {
            return dict;
        }

        @Override
        public Object invoke(BasicCommand command) throws CommandInvocationException {
            DictionaryCommand dc =  dict.findCommand(command);
            if (dc == null) {
                throw new CommandInvocationException("Error: No handler found for command %s with %d arguments", command.getCommand(), command.getArgumentCount());
            }
            return invoke(target, methods.get(dc), command);
        }

        private void add(DictionaryCommand dc, Method method) {
            methods.put(dc,method);
        }

        private Object invoke(Object target, Method method, BasicCommand command) throws CommandInvocationException {

            try {
                RawCommand rawCommand;
                if (command instanceof RawCommand) {
                    rawCommand = (RawCommand) command;
                } else if (command instanceof TokenizedCommand) {
                    rawCommand = convertToRaw((TokenizedCommand) command, method);
                } else {
                    throw new CommandInvocationException("Error: Unknown type of command " + command.getClass().getName());
                }
                return method.invoke(target, rawCommand.getArguments());
            } catch (IllegalAccessException | IllegalArgumentException ex) {
                throw new CommandInvocationException("Error: Can't invoke command", ex);
            } catch (InvocationTargetException ex) {
                throw new CommandInvocationException(ex);
            }
        }

        private RawCommand convertToRaw(TokenizedCommand tokenizedCommand, Method method) throws CommandInvocationException {
            Class<?>[] parameterTypes = method.getParameterTypes();
            List<Object> args = new ArrayList(parameterTypes.length);
            boolean varArgs = method.isVarArgs();
            for (int i = 0; i < parameterTypes.length; i++) {
                if (varArgs && i == parameterTypes.length - 1) {
                    Class varClass = parameterTypes[i];
                    Class elemClass = varClass.getComponentType();
                    Object theArray = Array.newInstance(elemClass, tokenizedCommand.getArgumentCount() - parameterTypes.length + 1);
                    for (int j = 0; j < Array.getLength(theArray); j++) {
                        Array.set(theArray, j, engine.convertArgToType(tokenizedCommand.getArgument(i + j), elemClass));
                    }
                    args.add(theArray);
                } else {
                    args.add(engine.convertArgToType(tokenizedCommand.getArgument(i), parameterTypes[i]));
                }
            }
            return new RawCommand(tokenizedCommand.getCommand(), args);
        }
    }
}
