package org.lsst.ccs.command;

import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;

/**
 * Provides help based on information from a command dictionary.
 *
 * @author tonyj
 */
public class HelpGenerator {

    private final Dictionary dict;
    private final PrintWriter out;

    /**
     * Create a HelpGenerator
     *
     * @param out Where the output of the help commands should be sent
     * @param dict The dictionary used to provide help
     */
    public HelpGenerator(PrintWriter out, Dictionary dict) {
        this.out = out;
        this.dict = dict;
    }

    @Command(description = "List available commands")
    public void help() {
        List<DictionaryCommand> sorted = new ArrayList<>();
        for (DictionaryCommand def : dict) {
            sorted.add(def);
        }
        Collections.sort(sorted, new CommandDefinitionComparator());
        for (DictionaryCommand def : sorted) {
            helpForCommand(def);
        }
    }

    @Command(description = "Show help for a single command")
    public void help(@Argument(name = "command") String command) {
        boolean foundCommand = false;
        for (DictionaryCommand def : dict) {
            if (def.getCommandName().equals(command)) {
                foundCommand = true;
                helpForCommand(def);
                int maxNameLeng = 0, maxTypeLeng = 0;
                for (DictionaryParameter param : def.getParams()) {
                    int leng = param.getName().length();
                    maxNameLeng = (leng > maxNameLeng) ? leng : maxNameLeng;
                    leng = param.getSimpleType().length();
                    maxTypeLeng = (leng > maxTypeLeng) ? leng : maxTypeLeng;
                }
                String spaces = String.format("%80s", "");
                for (DictionaryParameter param : def.getParams()) {
                    String name = param.getName(), type = param.getSimpleType();
                    out.printf("    %s%s   %s%s   %s\n", name,
                               spaces.substring(0, maxNameLeng - name.length()),
                               type,
                               spaces.substring(0, maxTypeLeng - type.length()),
                               param.getDescription());
                }
            }
        }
        if (!foundCommand) throw new IllegalArgumentException("No help found for "+command);
    }

    private void helpForCommand(DictionaryCommand def) {
        StringBuilder builder = new StringBuilder();
        builder.append(def.getCommandName());
        for (DictionaryParameter param : def.getParams()) {
            builder.append(' ').append(param.getName());
        }
        if (def.isVarArgs()) {
            builder.append("...");
        }
        out.printf(builder.length()<=30 ? "%-30s %s\n" : "%s\n                               %s\n", 
                builder, def.getDescription());
        if (def.getAliases().length>0) {
            out.printf("    aliases:");
            for (String alias : def.getAliases()) out.printf(" %s",alias);
            out.println();
        }
    }

    /**
     * A comparator used for putting commands into alphabetical order
     */
    private static class CommandDefinitionComparator implements Comparator<DictionaryCommand> {

        @Override
        public int compare(DictionaryCommand o1, DictionaryCommand o2) {
            return o1.getCommandName().compareTo(o2.getCommandName());
        }
    }
}
