package org.lsst.ccs.command;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.utilities.tracers.Tracer;

import java.lang.reflect.Method;
import java.util.ArrayList;

/**
 * Builds a command dictionary for a given class using annotations on the
 * methods of the class.
 *
 * @author tonyj
 */
class DictionaryBuilder {

    static {
        assert Tracer.version("$Rev$", DictionaryBuilder.class, "org-lsst-ccs-command-core");
    }

    /**
     * Build a command dictionary for a class using annotations on the methods
     * of the class.
     *
     * @param targetClass The class for which the dictionary should be built
     * @return The dictionary created.
     */
    Dictionary build(Class targetClass) {
        CommandDictionaryImplementation dict = new CommandDictionaryImplementation();
        for (Method method : targetClass.getMethods()) {
            Command annotation = method.getAnnotation(Command.class);
            if (annotation != null) {
                dict.add(new DictionaryCommand(method, annotation));
            }
        }
        // FIXME: Should this throw an exception if no annotations are found, or just and 
        // empty dictionary as we do now?
        return dict;
    }

    private static class CommandDictionaryImplementation extends ArrayList<DictionaryCommand> implements Dictionary {

        @Override
        public boolean containsCommand(BasicCommand tc) {
            return findCommand(tc) != -1;
        }

        @Override
        public int findCommand(BasicCommand tc) {
            return findCommand(tc.getCommand(), tc.getArgumentCount());
        }

        @Override
        public boolean containsCommand(String command, int argumentCount) {
            return findCommand(command, argumentCount) != -1;
        }

        @Override
        public int findCommand(String command, int argumentCount) {
            int index = 0;
            for (DictionaryCommand def : this) {
                if (def.getCommandName().equals(command)
                        && (argumentCount == def.getParams().length
                        || (argumentCount > def.getParams().length && def.isVarArgs()))) {
                    return index;
                }
                for (String alias : def.getAliases()) {
                    if (alias.equals(command)
                            && (argumentCount == def.getParams().length
                            || (argumentCount > def.getParams().length && def.isVarArgs()))) {
                        return index;
                    }
                }
                index++;
            }
            return -1;
        }
    }
}
