package org.lsst.ccs.command.demo.main;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JPanel;
import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.lsst.ccs.command.CommandSetBuilder;
import org.lsst.ccs.command.demo.ExtendedDemoCommands;
import org.lsst.ccs.command.demo.remote.jgroups.JGroupsCommandClient;
import org.lsst.ccs.command.demo.remote.jgroups.JGroupsCommandServer;
import org.lsst.ccs.demo.shell.SwingShell;
import org.lsst.ccs.shell.JLineShell;

/**
 * This main class acts as a steering class for the demos.
 *
 * @author tonyj
 */
public class Main {

    public static void main(String[] args) throws Exception {

        Options mainOptions = new Options();
        mainOptions.addOption("h", "help", false, "Print the help message");

        mainOptions.addOption("m", "mode", true, "The MODE in which the shell operates.\n"
                + "\tshell -- run the interactive shell locally"
                + "\tserver -- run a distributed command server"
                + "\tclient -- run a distributed command client"
                + "\tgui -- an interactive gui, the default");
        mainOptions.getOption("mode").setArgName("MODE");

        CommandLineParser parser = new BasicParser();
        CommandLine line = parser.parse(mainOptions, args, true);

        boolean printHelp = false;

        if (line.hasOption("help")) {
            printHelp = true;
        } else {
            String mode = line.getOptionValue("mode");

            Main main = new Main();
            if (mode != null) {
                switch (mode) {
                    case "shell":
                        Main.runShell();
                        break;
                    case "client":
                        Main.runClient();
                        break;
                    case "server":
                        Main.runServer();
                        break;
                    case "gui":
                        main.runGui();
                        break;
                    default:
                        printHelp = true;
                }
            } else {
                main.runGui();
            }
        }

        if (printHelp) {
            HelpFormatter formatter = new HelpFormatter();
            formatter.printHelp(100, "Command Demo", "", mainOptions, "", true);
        }
    }

    private void runGui() throws IOException {
        Main.GUIPanel gui = new Main.GUIPanel();
        JFrame frame = new JFrame("CCS Command Demo");
        frame.setContentPane(gui);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        frame.setLocationByPlatform(true);
        frame.pack();
        frame.setVisible(true);
    }

    private static void runShell() throws IOException, Exception {
        CommandSetBuilder builder = new CommandSetBuilder();
        JLineShell shell = new JLineShell(builder.buildCommandSet(new ExtendedDemoCommands()));
        shell.run();
    }

    private static void runClient() throws Exception {
        try (JGroupsCommandClient client = new JGroupsCommandClient()) {
            JLineShell shell = new JLineShell(client);
            shell.run();
        }
    }

    private static void runServer() throws Exception {
        CommandSetBuilder builder = new CommandSetBuilder();
        JGroupsCommandServer server = new JGroupsCommandServer(builder.buildCommandSet(new ExtendedDemoCommands()));
    }

    private static class GUIPanel extends JPanel implements ActionListener {

        public GUIPanel() {
            addButton(new JButton("Open Local Shell"), "shell");
            addButton(new JButton("Open Distributed command client"), "client");
            addButton(new JButton("Run Distributed command server"), "server");
        }

        @Override
        public void actionPerformed(ActionEvent e) {
            try {
                String command = e.getActionCommand();
                switch (command) {
                    case "shell":
                        runShell();
                        break;
                    case "client":
                        runClient();
                        break;
                    case "server":
                        runServer();
                        break;
                }
            } catch (Exception ex) {
                Logger.getLogger(Main.class.getName()).log(Level.SEVERE, null, ex);
            }

        }

        private void runShell() throws IOException {
            CommandSetBuilder builder = new CommandSetBuilder();
            final SwingShell swingShell = new SwingShell(builder.buildCommandSet(new ExtendedDemoCommands()), "Swing Shell");
            Thread t = new Thread() {
                @Override
                public void run() {
                    try {
                        swingShell.run();
                    } catch (IOException ex) {
                        Logger.getLogger(Main.class.getName()).log(Level.SEVERE, null, ex);
                    }
                }
            };
            t.start();
        }

        private void runClient() throws Exception {
            JGroupsCommandClient client = new JGroupsCommandClient();
            final SwingShell swingShell = new SwingShell(client, "Remote Swing Shell");
            Thread t = new Thread() {
                @Override
                public void run() {
                    try {
                        swingShell.run();
                    } catch (IOException ex) {
                        Logger.getLogger(Main.class.getName()).log(Level.SEVERE, null, ex);
                    }
                }
            };
            t.start();
        }

        private void runServer() {

        }

        private void addButton(JButton jButton, String command) {
            add(jButton);
            jButton.setActionCommand(command);
            jButton.addActionListener(this);
        }
    }
}

