"""
FCS set filter sequence
=======================

"""
from org.lsst.ccs.scripting import CCS
from org.lsst.ccs.command import *
from java.time import Duration
from java.lang import Exception
from time import sleep, time
from random import shuffle

CCS.setThrowExceptions(True)

class SetFilterSequence(object):
    name = "fcs"

    def __init__(self, n_sequences, handoff_only, random, wait_time=60):
        self.n_seq = n_sequences
        self.handoff_only = handoff_only
        self.random = random
        self.wait_time = wait_time

        self.subsystem = CCS.attachSubsystem(self.name, 1)
        print("Connected to fcs")
        print(self.subsystem.sendSynchCommand("getCCSVersions"))

        self.config = self.subsystem.sendAsynchCommand("getConfigurationInfo").get()
        self.socket2filter = dict(self.subsystem.sendSynchCommand("getSocketsToAvailableFiltersID"))
        self.timeout = Duration.ofMinutes(3)

    def setFilter(self, socket_name):
        filter_id = self.socket2filter[socket_name]
        print("Switching to filter %d on %s.." % (filter_id, socket_name))

        t0 = time()

        if self.handoff_only:
            self.subsystem.sendSynchCommand(self.timeout, "setFilterAtHandoff", filter_id)
        else:
            self.subsystem.sendSynchCommand(self.timeout, "setFilter", filter_id)

        elapsed_time = time() - t0

        print("\t..took %dmin%ds" % (elapsed_time // 60, elapsed_time % 60))

    def setFilterSequence(self):
        socket_order = list(self.socket2filter.keys())
        if self.random:
            shuffle(socket_order)
        t0 = time()
        for socket in socket_order:
            self.setFilter(socket)
            print("Waiting %s seconds for cooldown.." % self.wait_time)
            sleep(self.wait_time)

        elapsed_time = time() - t0

        print("=> sequence took %dmin%ds\n" % (elapsed_time // 60, elapsed_time % 60))

    def run(self):
        """Execute the sequence"""
        print(__doc__)

        for i in range(self.n_seq):
            print("Starting sequence %d/%d" % (i + 1, self.n_seq))
            self.setFilterSequence()

        print("All sequences ended, setting no filter ONLINE")
        self.subsystem.sendSynchCommand(self.timeout, "setFilter", 0) 

        print("Done")
    

if __name__ == "__main__":
    import argparse

    parser = argparse.ArgumentParser()
    parser.add_argument(
        "-n", "--n_sequences", 
        type=int, default=1, 
        help="Number of full sequences executed (default 1)")
    parser.add_argument(
        "--handoff", 
        action='store_true', 
        help="Filter never go past HANDOFF position")
    parser.add_argument(
        "-r", "--random", 
        action='store_true', 
        help="Randomises the sockets order between exchanges")
    parser.add_argument(
        "-w", "--wait",
        type=int, default=120,
        help="Wait time between sequences in seconds (default 120)"
    )

    args = parser.parse_args()

    SetFilterSequence(args.n_sequences, args.handoff, args.random, args.wait).run()
