/*
 * Decompiled with CFR 0.152.
 */
package org.lsst.ccs.subsystem.imagehandling;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.Serializable;
import java.nio.file.Files;
import java.nio.file.attribute.FileAttribute;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.imagehandling.ImageHandlingConfig;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import org.lsst.ccs.subsystem.imagehandling.imagedb.ImageFileDatabase;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.utilities.location.SensorLocation;

public class CommandExecutor {
    private static final Logger LOG = Logger.getLogger(CommandExecutor.class.getName());
    private static final Pattern TELEMETRY_PATTERN = Pattern.compile("Telemetry\\:\\s*(.*)\\:\\s*(.*)");
    private static final Pattern SLOT_PATTERN = Pattern.compile(".*_S(.*)\\.fits");
    private final ImageHandlingConfig config;
    private final AtomicInteger runningProcesses = new AtomicInteger();
    private final Agent agent;
    private final ImageFileDatabase ifd;

    public CommandExecutor(ImageHandlingConfig config, Agent agent, ImageFileDatabase ifd) {
        this.config = config;
        this.agent = agent;
        this.ifd = ifd;
    }

    void executeFitsFileList(FileList fileList, Map<String, String> env, Location location, ImageName imageName) {
        for (File file : fileList) {
            Matcher matcher = SLOT_PATTERN.matcher(file.getName());
            if (matcher.matches()) {
                String slot = "S" + matcher.group(1);
                SensorLocation sensorLocation = new SensorLocation(location, Integer.parseInt(slot.substring(2, 3)));
                this.executeFitsFile(file, env, sensorLocation, imageName, slot);
                continue;
            }
            LOG.log(Level.WARNING, "Ignored malformed file name {0}", file.getName());
        }
    }

    void executeFitsFile(File fitsFile, Map<String, String> env, SensorLocation sensorLocation, ImageName imageName, String slot) {
        List<String> commands = this.config.getCommands();
        LinkedHashMap<String, String> newEnv = new LinkedHashMap<String, String>(env);
        newEnv.put("FITSFILE", fitsFile.getAbsolutePath());
        newEnv.put("SLOT", slot);
        newEnv.put("FILETYPE", "FITS");
        int fileId = this.registerFileInDatabase(imageName, sensorLocation, fitsFile, "FITS");
        CompletableFuture<File> future = this.executeFile(fitsFile, newEnv, commands);
        future.thenAccept(logFile -> {
            String raft = newEnv.getOrDefault("RAFT", "R??");
            KeyValueDataList keyValueDataList = this.scanFileForTelemetry((File)logFile, "scripts/" + raft + "/" + slot);
            if (!keyValueDataList.getListOfKeyValueData().isEmpty()) {
                this.agent.publishSubsystemDataOnStatusBus((KeyValueData)keyValueDataList);
            }
            this.registerOperations(fileId, (File)logFile, fitsFile, false);
        });
    }

    private int registerFileInDatabase(ImageName imageName, SensorLocation sl, File file, String fileType) {
        int fileId = 0;
        if (this.ifd != null) {
            try {
                fileId = this.ifd.insertFile(imageName, sl, fileType, file.getAbsolutePath());
            }
            catch (SQLException x) {
                LOG.log(Level.WARNING, "Could not update image database for file: " + file, x);
            }
        }
        return fileId;
    }

    private void registerOperations(int fileId, File logFile, File file, boolean ins_update) {
        if (this.ifd != null && fileId != 0) {
            try {
                List<ImageFileDatabase.Operation> operations = this.scanFileForOperations(logFile, fileId);
                this.ifd.insertFileOperations(operations, ins_update);
            }
            catch (IOException | SQLException ex) {
                LOG.log(Level.WARNING, "Could not update image operations database for file: " + file, ex);
            }
        }
    }

    private CompletableFuture<File> executeFile(File file, Map<String, String> env, List<String> commands) {
        List<String> command = List.of("/bin/bash", "-c", commands.stream().collect(Collectors.joining(";")));
        String date = env.getOrDefault("DATE", "UNKNOWN");
        File logFileDir = new File(this.config.getLogDirectory(), date);
        File logFile = new File(logFileDir, file.getName() + ".log");
        ProcessBuilder processBuilder = new ProcessBuilder(command);
        processBuilder.environment().putAll(this.config.getEnvironment());
        processBuilder.environment().putAll(env);
        processBuilder.environment().put("LOGFILE", logFile.getAbsolutePath());
        processBuilder.environment().put("FILE", file.getAbsolutePath());
        processBuilder.directory(this.config.getCurrentDirectory());
        processBuilder.redirectErrorStream(true);
        processBuilder.redirectOutput(logFile);
        try {
            CompletableFuture<File> result = new CompletableFuture<File>();
            Files.createDirectories(logFileDir.toPath(), new FileAttribute[0]);
            int running = this.runningProcesses.incrementAndGet();
            processBuilder.environment().put("RUNNING", String.valueOf(running));
            Process process = processBuilder.start();
            LOG.log(Level.INFO, () -> String.format("Process for file %s started (%d running) on thread %s", file, running, Thread.currentThread().getName()));
            process.onExit().thenAccept(p -> {
                this.runningProcesses.decrementAndGet();
                int rc = p.exitValue();
                if (rc != 0) {
                    String message = String.format("Process for file %s terminated with non-zero return code %d", file, rc);
                    LOG.log(Level.WARNING, message);
                    result.completeExceptionally(new RuntimeException(message));
                } else {
                    LOG.log(Level.INFO, () -> String.format("Process for file %s complete", file));
                    result.complete(logFile);
                }
            });
            return result;
        }
        catch (IOException x) {
            LOG.log(Level.WARNING, "Unable to create process for file " + file, x);
            return CompletableFuture.failedFuture(x);
        }
    }

    private KeyValueDataList scanFileForTelemetry(File logFile, String root) {
        KeyValueDataList kvdl = new KeyValueDataList("");
        try (BufferedReader in = Files.newBufferedReader(logFile.toPath());){
            String line;
            while ((line = in.readLine()) != null) {
                Matcher matcher = TELEMETRY_PATTERN.matcher(line);
                if (!matcher.matches()) continue;
                String key = matcher.group(1);
                String valueStr = matcher.group(2);
                double value = Double.NaN;
                try {
                    value = Double.parseDouble(valueStr);
                }
                catch (NumberFormatException nfe) {
                    LOG.log(Level.WARNING, "Error parsing telemetry from command output, {0}", valueStr);
                }
                if (key.startsWith("/")) {
                    kvdl.addData(key, (Serializable)Double.valueOf(value));
                    continue;
                }
                kvdl.addData(root + "/" + key, (Serializable)Double.valueOf(value));
            }
        }
        catch (IOException x) {
            LOG.log(Level.WARNING, "IO Error scanning log file: " + logFile, x);
        }
        return kvdl;
    }

    CompletableFuture<File> executeAdditional(File file, ImageName obsId, String fileType, Map<String, String> env) {
        Object fileTypeForDatabase = fileType;
        if ("shutterMotionProfile".equals(fileType)) {
            fileTypeForDatabase = file.toString().contains("Open") ? (String)fileTypeForDatabase + "Open" : (String)fileTypeForDatabase + "Shut";
        }
        int fileId = this.registerFileInDatabase(obsId, null, file, (String)fileTypeForDatabase);
        CompletableFuture<File> future = this.executeFile(file, env, this.config.getAdditionalFileCommands());
        return future.thenApply(logFile -> {
            this.registerOperations(fileId, (File)logFile, file, false);
            return logFile;
        });
    }

    CompletableFuture<File> executeMissingFile(List<String> commands, File fitsFile, int fileId, Map<String, String> env) {
        CompletableFuture<File> future = this.executeFile(fitsFile, env, commands);
        return future.thenApply(logFile -> {
            this.registerOperations(fileId, (File)logFile, fitsFile, true);
            return logFile;
        });
    }

    public List<ImageFileDatabase.Operation> scanFileForOperations(File logFile, int fileId) throws IOException {
        long timeStampStart = 0L;
        String currentOperation = null;
        StringBuilder logContent = new StringBuilder();
        ArrayList<ImageFileDatabase.Operation> result = new ArrayList<ImageFileDatabase.Operation>();
        try (BufferedReader br = new BufferedReader(new FileReader(logFile));){
            String readLine;
            boolean inParseBlock = false;
            while ((readLine = br.readLine()) != null) {
                String[] tokens = readLine.split(" ");
                boolean start = tokens[0].equals("Start:");
                boolean end = tokens[0].equals("End:");
                if (inParseBlock && start) {
                    LOG.log(Level.WARNING, "Error in log file - 2 consecutive Starts encountered \n");
                }
                if (start && !inParseBlock) {
                    inParseBlock = true;
                    currentOperation = tokens[1];
                    timeStampStart = Long.parseLong(tokens[2]);
                    logContent.setLength(0);
                }
                if (!end && !start && inParseBlock) {
                    logContent.append(readLine).append('\n');
                    continue;
                }
                if (!end || !tokens[1].equals(currentOperation) || !inParseBlock) continue;
                inParseBlock = false;
                long timeStampStop = Long.parseLong(tokens[3]);
                Float duration = Float.valueOf((float)(timeStampStop - timeStampStart) / 1000.0f);
                int rc = Integer.parseInt(tokens[2]);
                boolean success = rc == 0;
                String reasons = null;
                if (!success) {
                    reasons = String.format("%.256s", logContent);
                } else if (end && !tokens[1].equals(currentOperation) && inParseBlock) {
                    LOG.log(Level.WARNING, "Error in log file - current operation at End doesn't match with Start \n");
                }
                result.add(new ImageFileDatabase.Operation(fileId, currentOperation, rc, duration.floatValue(), success, reasons));
            }
        }
        return result;
    }
}

