"""
FCS single filter sequence
==========================
Test a sequence of setFilter commands with a single filter in the camera.
The order of the sockets is picked up randomly.

"""
from org.lsst.ccs.scripting import CCS
from org.lsst.ccs.command import *
from java.time import Duration
from java.lang import Exception
from time import sleep, time
from random import choice

CCS.setThrowExceptions(True)


class SingleFilterSequence(object):
    name = "fcs"

    def __init__(self, filter_id, n_sequences, wait_time=120):
        self.filter_id = filter_id
        self.n_seq = n_sequences
        self.wait_time = wait_time

        self.fcs = CCS.attachSubsystem(self.name, 1)
        self.autochanger = CCS.attachSubsystem("%s/%s" % (self.name, "autochanger"))
        self.carousel = CCS.attachSubsystem("%s/%s" % (self.name, "carousel"))
        print("Connected to %s" % self.name)

        self.timeout = Duration.ofMinutes(3)

    def pick_next_socket(self):
        socket_list = range(1, 6)
        next_socket = choice(socket_list)
        previous_socket = int(self.carousel.sendSynchCommand("getSocketAtStandbyID"))
        while next_socket == previous_socket:
            next_socket = choice(socket_list)
        
        return next_socket

    def set_filter(self, filter_id):
        self.fcs.sendSynchCommand(self.timeout, "setFilter", filter_id)

    def set_no_filter(self):
        self.set_filter(0)

    def rotate_to_socket(self, socket_id):
        self.carousel.sendSynchCommand("rotateSocketToStandby", "socket%d" % socket_id)

    def run(self):
        """Execute the sequence"""
        print(__doc__)
        socket_id = 0
        # Make sure there the autochanger is empty first
        if self.autochanger.sendSynchCommand("isHoldingFilter"):
            self.fcs.sendSynchCommand("storeFilterOnCarousel")
        # Make sure the autochanger trucks are aligned
        self.autochanger.sendSynchCommand("alignFollower")

        for i in range(self.n_seq):
            t0 = time()
            print("Starting sequence %d/%d" % (i + 1, self.n_seq))
            
            self.set_filter(self.filter_id)
            
            t1 = time()
            print("+ set filter in [%dmin%ds]" % ((t1 - t0) // 60, (t1 - t0) % 60))
            
            socket_id = self.pick_next_socket()
            self.rotate_to_socket(socket_id)
            
            t2 = time()
            print("+ rotate to socket [%dmin%ds]" % ((t2 - t1) // 60, (t2 - t1) % 60))
            
            self.set_no_filter()
            
            t3 = time()
            print("+ set no filter in [%dmin%ds]" % ((t3 - t2) // 60, (t3 - t2) % 60))

            elapsed_time = time() - t0
            
            print("=> sequence took %dmin%ds\n" % (elapsed_time // 60, elapsed_time % 60))

            # one min pause for motor cooldown
            if i != (self.n_seq - 1):
                print("Pausing for 2 minutes for FES cooldown")
                sleep(self.wait_time)

        print("All sequences ended, setting no filter ONLINE")
        self.fcs.sendSynchCommand(self.timeout, "setFilter", 0)

        print("Done")
    

if __name__ == "__main__":
    import argparse

    parser = argparse.ArgumentParser()
    parser.add_argument(
        "-i", "--filter_id",
        type=int, required=True,
        help="Filter ID to be set")
    parser.add_argument(
        "-n", "--n_sequences", 
        type=int, default=1, 
        help="Number of full sequences executed (default 1)")
    parser.add_argument(
        "-w", "--wait",
        type=int, default=120,
        help="Wait time between sequences in seconds (default 120)"
    )

    args = parser.parse_args()

    SingleFilterSequence(args.filter_id, args.n_sequences, args.wait).run()
