package org.lsst.ccs.bus.states;

/**
 * Marker interface to distinguish internal states of an {@code Agent}. Classes
 * implementing this interface should extend {@code Enum}.
 * 
 * Static methods of this interface check constraints of the Agent state machine,
 * as documented at https://confluence.slac.stanford.edu/display/LSSTCAM/State+Diagrams.
 *
 * @author onoprien
 */
public interface AgentState {

    /**
     * Returns {@code true} if Agent internal states in the specified bundle are consistent.
     * 
     * @param bundle state bundle to check
     * @return {@code false} if Agent internal state in the specified bundle violates state machine constraints
     */
    static public boolean isLegal(StateBundle bundle) {

        AlertState alertState = (AlertState) bundle.getState(AlertState.class);
        OperationalState operationalState = (OperationalState) bundle.getState(OperationalState.class);
        ConfigurationState configurationState = (ConfigurationState) bundle.getState(ConfigurationState.class);

        if (alertState == AlertState.ALARM && operationalState != OperationalState.ENGINEERING_FAULT) {
            return false;
        }
        
        if (operationalState == OperationalState.NORMAL && configurationState == ConfigurationState.DIRTY){
            return false;
        }
        
        

        return true;
    }

    /**
     * Returns {@code true} if the transition between Agent internal states specified 
     * by the two bundles is allowed.
     * 
     * @param before state before transition
     * @param after state after transition
     * @return {@code false} if the transition violates state machine constraints
     */
    static public boolean isLegal(StateBundle before, StateBundle after) {

        if (!isLegal(after)) {
            return false;
        }

        OperationalState operationalBefore = (OperationalState) before.getState(OperationalState.class);
        OperationalState operationalAfter = (OperationalState) after.getState(OperationalState.class);
        PhaseState phaseBefore = (PhaseState) before.getState(PhaseState.class);
        PhaseState phaseAfter = (PhaseState) after.getState(PhaseState.class);

        
        if (phaseBefore != phaseAfter) {
            if (phaseBefore == null) {
                if (phaseAfter != PhaseState.INITIALIZING) return false;
            } else {
                switch (phaseBefore) {
                    case OFF_LINE:
                        return false;
                    case INITIALIZING:
                        if (!(phaseAfter == PhaseState.OPERATIONAL || phaseAfter == PhaseState.CLOSING)) return false;
                        break;
                    case OPERATIONAL:
                        if (!(phaseAfter == PhaseState.CLOSING)) return false;
                        break;
                    case CLOSING:
                        if (!(phaseAfter == PhaseState.OFF_LINE)) return false;
                        break;
                }
            }
        }
        
        if (operationalBefore != operationalAfter) {
            if (operationalBefore == null || operationalBefore == OperationalState.ENGINEERING_FAULT) {
                if (operationalAfter == OperationalState.NORMAL) return false;
            }
        }
        
        return true;
    }

}
