package org.lsst.ccs.config;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.lsst.ccs.bootstrap.SubstitutionTokenUtils;

/**
 * Class that describes a category tag, i.e. the particular configuration tag
 * that has been loaded for a given category.
 * 
 * The syntax for a category tag is: "categoryName:tag1|tag2|tag3"
 * where the pipe symbol "|" is used to separate the various tags that have been
 * loaded for this category.
 * 
 * Please note the following special cases:
 * 
 *  - "categoryName" means that the default tag ("") has been loaded for the given category
 *  - ":" means that the default tag ("") was loaded for the default category ("")
 *  - ":tag1" means that "tag1" was loaded for the default category ("")
 *  - "categoryName:tag1|tag2" means that "tag1" and "tag2" (in this order) have been loaded for category "categoryName"
 * 
 * @author The LSST CCS Team
 */
public class CategoryDescription implements Serializable {

    private static final long serialVersionUID = 740497758514140401L;
    
    private static final Logger LOG = Logger.getLogger(CategoryDescription.class.getName());
    
    private final String categoryName;
    private final String defaultSource;
    private final List<SingleCategoryTag> singleTags = new ArrayList<>();
    volatile boolean hasChanges = false;
    
    @Deprecated
    public static final String DEFAULT_CONFIG_NAME = "";
    public static final String DEFAULT_INITIAL_CONFIG_NAME = "defaultInitial";
    @Deprecated
    public static final String DEFAULT_CAT = "";
    public static final String DEFAULT_V_STRING = "d";
    public static final int DEFAULT_VERSION = -1;
    public static final int LATEST_VERSION = -2;
    public static final int UNDEF_VERSION = -3;
    
    private static final Pattern CATEGORY_TAG_PATTERN = Pattern.compile("((?<category>[a-zA-Z0-9]+)[:]?)(?<tags>.*?)(?<hasChanged>\\*?)");
    private static final Pattern TAG_PATTERN = Pattern.compile("((?<source>.*?)[/]{1})?(?<tag>[a-zA-Z0-9-]+)([(]{1}?(((?<version>[a-zA-Z0-9]*))[=]?(?<resolvedVersion>.*))[)]{1})?");
    private static final Pattern TAG_SYNTAX = Pattern.compile("^[a-zA-Z0-9-]+$");
        
    public CategoryDescription(String categoryName, String defaultSource) {
        this.categoryName = categoryName;
        this.defaultSource = defaultSource;
    }

    public String getCategoryName() {
        return categoryName;
    }
    
    @Deprecated
    public boolean containsSingleTag(String tag, String source) {
        return CategoryDescription.this.containsSingleTag(new SingleCategoryTag(source, tag, categoryName));
    }

    public boolean containsSingleTag(SingleCategoryTag singleCategoryTag) {
        return singleTags.contains(singleCategoryTag);
    }
    
    public SingleCategoryTag getSingleTag(SingleCategoryTag singleCategoryTag) {
        if ( ! containsSingleTag(singleCategoryTag) ) {
            throw new IllegalArgumentException("Tag "+singleCategoryTag+" does not belong to "+this);
        }
        return singleTags.get(singleTags.indexOf(singleCategoryTag));
    }

    public String getSingleTagVersion(SingleCategoryTag singleCategoryTag) {
        if ( ! singleTags.contains(singleCategoryTag) ) {
            throw new IllegalArgumentException("Sourced tag \""+singleCategoryTag+"\" does not exist for this CategoryTag");
        }
        return singleTags.get(singleTags.indexOf(singleCategoryTag)).getVersion();
    }

    @Override
    public String toString() {
        return convertToString(false);
    }
    
    @Override
    public int hashCode() {
        int hash = 23;
        hash = 33 * hash + Objects.hashCode(this.singleTags)+45*categoryName.hashCode()+Objects.hashCode(hasChanges);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final CategoryDescription other = (CategoryDescription) obj;
        if ( hasChanges != other.hasChanges ) {
            return false;
        }
        if ( !categoryName.equals(other.categoryName) ) {
            return false;
        }
        return singleTags.equals(other.singleTags);
    }

    public String convertToString(boolean showDefaultSource) {
        StringBuilder sb = new StringBuilder();
        sb.append(getCategoryName()).append(":");
        int nTags = getNumberOfTags();
        for(SingleCategoryTag sourcedTag : getSingleCategoryTags()) {
            nTags--;
            sb.append(sourcedTag.convertToString(false, showDefaultSource));
            if (nTags > 0) {
                sb.append("|");
            }
        }
        if ( hasChanges ) {
            sb.append("*");
        }
        return sb.toString();        
    }

    public void setHasChanges(boolean hasChanges) {
        this.hasChanges = hasChanges;
    }
    
    public boolean hasChanges() {
        return hasChanges;
    }
        
    public List<SingleCategoryTag> getSingleCategoryTags() {
        return new ArrayList<>(singleTags);
    }
    
    public SingleCategoryTag getTopMostTag() {
        return singleTags.get(singleTags.size()-1);
    }
    
    public void addOrUpdadateSingleTagVersion(String source, String tag, String version) {
        SingleCategoryTag singleTag = new SingleCategoryTag(source, tag, categoryName, version);
        if ( defaultSource != null ) {
            singleTag.setDefaultSource(defaultSource);
        }
        addSingleTagVersion(singleTag,version,true);
    }

    public void addOrUpdadateSingleTag(SingleCategoryTag singleCategoryTag) {
        SingleCategoryTag singleTag = new SingleCategoryTag(singleCategoryTag);
        if ( defaultSource != null ) {
            singleTag.setDefaultSourceIfAbsent(defaultSource);
        }
        addSingleTagVersion(singleTag,singleTag.getResolvedVersion(),true);
    }

    public void addSingleTagVersion(SingleCategoryTag singleTag, String version, boolean update) {
        if ( singleTags.contains(singleTag) && !update ) {
            throw new RuntimeException("Tag: "+singleTag.getTag()+" for source: "+singleTag.getSource()+" already exists for this CategoryTag");            
        }
        SingleCategoryTag newTag = new SingleCategoryTag(singleTag,version);
        if ( defaultSource != null ) {
            newTag.setDefaultSourceIfAbsent(defaultSource);
        }
        int ind = singleTags.indexOf(newTag);
        if ( ind < 0 ) {
            singleTags.add(newTag);                    
        } else {
            singleTags.remove(ind);
            singleTags.add(ind, newTag);
        }        
    }

    String getDefaultSource() {
        return defaultSource;
    }

    
    int getNumberOfTags() {
        return singleTags.size();
    }
    
    public CategoryDescription merge(CategoryDescription categoryTag) {
        for(SingleCategoryTag singleTag : categoryTag.getSingleCategoryTags()) {
            addSingleTagVersion(singleTag, singleTag.getResolvedVersion(), true);
        }
        return this;
    }
    
    public static final CategoryDescription parseCategoryTagInput(String categoryTagInput) {
        return parseCategoryTagInput(categoryTagInput, null);
    }

    public static final CategoryDescription parseCategoryTagInput(String categoryTagInput, Function<String,String> defaultSourcesFunction) {
        categoryTagInput = SubstitutionTokenUtils.resolveSubstitutionTokens(categoryTagInput);
        Matcher m = CATEGORY_TAG_PATTERN.matcher(categoryTagInput);
        if (m.matches()) {
            String cat = m.group("category");
            if (cat == null) {
                throw new IllegalArgumentException("A Catogory must be specified. It cannot be empty.");
            }
            
            String defaultSource = null;
            if ( defaultSourcesFunction != null ) {
                defaultSource = defaultSourcesFunction.apply(cat);
            }
            if ( defaultSource == null ) {
                //Fallback to static map
                defaultSource = SingleCategoryTag.getDefaultSourceForCategory(cat);
            }
            
            CategoryDescription categoryTag = new CategoryDescription(cat, defaultSource);
            String allTags = m.group("tags");

            if (allTags != null && allTags.endsWith("|")) {
                throw new IllegalArgumentException("Tags cannot be empty, so a list of tags cannot end with \"|\".");
            }

            if (allTags != null) {
                String[] tags = allTags.split("\\|");

                boolean hasChanged = "*".equals(m.group("hasChanged"));
                categoryTag.setHasChanges(hasChanged);

                for (String t : tags) {
                    Matcher tagMatcher = TAG_PATTERN.matcher(t);
                    if (tagMatcher.matches()) {

                        String tag = tagMatcher.group("tag");
                        if (tag == null) {
                            throw new IllegalArgumentException("A tag cannot be empty: it must be specified.");
                        } else if (!tag.isEmpty()) {
                            if (!TAG_SYNTAX.matcher(tag).matches()) {
                                throw new IllegalArgumentException("Illegal syntax for tag: \"" + tag + "\". Only letters and digits are allowed for tag names.");
                            }
                        }

                        String source = tagMatcher.group("source");
                        if (source == null || source.isEmpty()) {
                            source = defaultSource;
                        }

                        String v = tagMatcher.group("version");
                        String ver = v;
                        if (v == null || v.isEmpty() ) {
                            ver = "";
                        }
                        
                        String resolvedVersion = tagMatcher.group("resolvedVersion");
                        if ( resolvedVersion == null || resolvedVersion.isEmpty() ) {
                            resolvedVersion = ver;
                        }
                        
                        categoryTag.addSingleTagVersion(new SingleCategoryTag(source, tag, cat,ver), resolvedVersion, false);
                    } else {
                        throw new IllegalArgumentException("Could not accept tag: " + t+". Allowed characters are [0-9][a-zA-Z] and dash \"-\".");
                        
                        
                    }
                }
            }
            return categoryTag;
        }
        throw new IllegalArgumentException("Could not parse a CategoryTag from input string: \"" + categoryTagInput + "\"");
    }
    
}
