package org.lsst.ccs.bus.messages;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;

/**
 * Base class for all status messages containing data that is private
 * to the sender Subsystem.
 * Because clients will not necessarily have the class definition for the embedded
 * object, this class relies on the encoding procedure of the embedded object.
 * 
 * @author LSST CCS Team
 */
public final class StatusSubsystemData extends StatusData {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = 43353823972423L;
    
    private final String key;

    private final Map<String, Serializable> attributes = new HashMap<>();

    /**
     * Build a StatusMessage from the provided Object.
     *
     * @param kvData The KeyValue object to be sent over the buses.
     */
    public StatusSubsystemData(KeyValueData kvData) {
        super(kvData);
        key = kvData.getKey();
        if ( kvData instanceof KeyValueDataList ) {
            attributes.putAll(((KeyValueDataList)kvData).getAttributes());
        }
    }
    
    /**
     * Get the Key of the embedded object.
     * If the embedded object is a KeyValueData, then the KeyValueData's key
     * is returned, otherwise the canonical name of the class of the embedded
     * object.
     * 
     * @return The key of the embedded object.
     */    
    public String getDataKey() {
        return key;        
    }

    /**
     * Get the embedded subsystem data.
     * 
     * @return The subsystem data embedded in this message.
     * @throws EmbeddedObjectDeserializationException if the embedded object
     * cannot be deserialized.
     * 
     */
    public KeyValueData getSubsystemData() {
        return (KeyValueData)getObject();
    }
    
    /**
     * Get all the data attributes.
     *
     * @return The data attributes.
     */
    public Map<String,Serializable> getDataAttributes() {
        return new HashMap<>(attributes);
    }

}
