package org.lsst.ccs.bus.messages;

import java.util.Arrays;
import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.command.BasicCommand;
import org.lsst.ccs.command.RawCommand;
import org.lsst.ccs.command.TokenizedCommand;

/**
 * Wrapper class to send a BasicCommand over the Command bus; the BasicCommand is either
 * provided as an object or created internally from a String representation: e.g. "doSomething 1 2".
 * 
 * @author LSST CCS Team
 * 
 */
public final class CommandRequest extends CommandMessage<BasicCommand, String> {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = 398321334356909L;
    
    /** The lock attached to this command request. */
    private AgentLock lock;
    
    /** A level that optionally overrides the lock level. */
    private int level;

    /**
     * Create a BasicCommandMessage from a string representation of a command: e.g. "doSomething 1 2".
     * @param destination The destination of this command.
     * @param command     The string representation of a command. A BasicCommand object will be created internally.
     */
    public CommandRequest(String destination, String command) {
        this(destination, command, (Object[]) null);
    }
    
    /**
     * Create a BasicCommandMessage from a string representation of a command: e.g. "doSomething 1 2".
     * @param destination The destination of this command.
     * @param command     The string representation of a command. A BasicCommand object will be created internally.
     * @param args        The array of arguments to the command.
     * 
     */
    public CommandRequest(String destination, String command, Object ... args) {
        this(destination, (args != null && args.length > 0) ? new RawCommand(command, Arrays.asList(args)) : new TokenizedCommand(command));
    }

    /**
     * Create a BasicCommandMessage from an existing BasicCommand.
     * @param destination The destination of this command.
     * @param command     The BasicCommand object to be wrapper.
     */
    public CommandRequest(String destination, BasicCommand command) {
        super(destination,java.util.UUID.randomUUID(),command);
    }
    
    /*
     * If there is ever a performance issue we can return null and skip encoding.
     */
    @Override
    protected String encodeObject(BasicCommand obj) {
        return obj.toString();
    }

    /**
     * Utility method to get the embedded BasicCommand.
     * This is equivalent to invoking the BusMessage getObject method.
     * 
     * @return The BasicCommand embedded in this CommandRequest.
     * @throws EmbeddedObjectDeserializationException if the embedded object
     * cannot be deserialized
     */
    public BasicCommand getBasicCommand() {
        return getObject();
    }
    
    /**
     * Get the lock associated to this command request.
     * @return a lock object.
     */
    public AgentLock getLock() {
        return lock;
    }
    
    /**
     * Get the level associated to this command request.
     * @return 
     */
    public int getLevel() {
        return level;
    }
    
    /**
     * Associate lock and level sender's information to this command request.
     * Implementation note : this makes the CommandRequest object mutable, and
     * potentially any arbitrary lock object could be set to this command
     * request. But the bus application layer will have the final word on which
     * lock will be set.
     *
     * @param lock
     * @param level
     */
    public void setLockAndLevel(AgentLock lock, int level) {
        this.lock = lock;
        this.level = level;
    }
    
    @Override
    public String toString() {
        return "CommandRequest to : "+getDestination()+" "+getBasicCommand();
    }

}
