package org.lsst.ccs.bus.data;

import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * A history of alerts for a specific alert ID, since the last clear.
 * An instance of this class contains a list of {@link RaisedAlertInstance}
 * objects created by calls to {@code addAlertInstance(...)} method.
 * 
 * This class is meant to be sent over the buses as part of a {@link RaisedAlertSummary} object.
 *
 * @author The LSST CCS Team.
 */
public final class RaisedAlertHistory implements Serializable {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = 423621057372456934L;

    private Alert latestAlert;
    private final List<RaisedAlertInstance> instances;
    private AlertState highestSeverity = AlertState.NOMINAL;
    private Alert highestSeverityAlert;
    private Acknowledgement ack;

    /**
     * The main constructor of the RaisedAlert.
     */
    public RaisedAlertHistory() {
        instances = new ArrayList<>(1);
    }

    /**
     * Copy constructor.
     * @param raisedAlertHistory The RaisedAlertHistory to copy.
     */
    public RaisedAlertHistory(RaisedAlertHistory raisedAlertHistory) {
        this();
        this.latestAlert = raisedAlertHistory.latestAlert;
        this.instances.addAll(raisedAlertHistory.instances);
        this.highestSeverity = raisedAlertHistory.highestSeverity;
        this.highestSeverityAlert = raisedAlertHistory.highestSeverityAlert;
        this.ack = raisedAlertHistory.ack;
    }
    
    /**
     * Get the original Alert that was raised.
     * 
     * @return The original Alert that was raised.
     */
    public Alert getLatestAlert() {
        return latestAlert;
    }
    
    /**
     * Add a new instance of this Alert.
     * 
     * @param severity The AlertState of the Alert.
     * @param alert The Alert that was raised.
     * @param ccsTimeStamp The CCS timestamp of when the Alert was raised.
     * @param cause The cause of the Alert.
     * @param count the number of alerts this instance stands for.
     */
    public void addAlertInstance(AlertState severity, Alert alert, CCSTimeStamp ccsTimeStamp, String cause, int count) {
        //REVIEW: This is currently assuming that the instances
        //are already added in the right order.
        //Should we sort this list based on the RaisedAlertInstance?
        if ( severity.compareTo(highestSeverity) > 0 ) {
            highestSeverity = severity;
            highestSeverityAlert = alert;
        }
        if (highestSeverityAlert == null) {
            highestSeverityAlert = alert;
        }
        instances.add(new RaisedAlertInstance(severity, ccsTimeStamp, cause, count));
        if ( instances.size() > 10 ) {
            for (int i = 9; i < instances.size(); i++ ) {
                instances.remove(0);
            }
        }
        if (severity != AlertState.NOMINAL) {
            ack = null;
        }
        latestAlert = alert;
    }
    
    /**
     * Acknowledge this alert.
     * @param message Message from the user who acknowledges this alert.
     * @param user User who acknowledges this alert.
     * @return True if the acknowledgement state changed as a result of this call.
     */
    public boolean addAcknowledgement(String message, String user) {
        if (ack == null) {
            ack = new Acknowledgement(message, user, this);
            return true;
        } else {
            return false;
        }
    }
    
    /**
     * Remove acknowledgement.
     * @return True if the acknowledgement state changed as a result of this call.
     */
    public boolean removeAcknowledgement() {
        if (ack == null) {
            return false;
        } else {
            ack = null;
            return true;
        }
    }
    
    /**
     * Get the number of times this Alert was raised.
     * 
     * @return the number of times the Alert was raised.
     */
    public int getNumberOfInstances() {
        return instances.size();

    }
    
    /**
     * Get the accumulated number of times this alert was raised.
     * @return the accumulated number of times this alert was raised.
     */
    public int getAccumulatedNumberOfInstances() {
        return instances.stream().collect(Collectors.summingInt(RaisedAlertInstance::getCount));
    }
    
    /**
     * Get the latest AlertState of this Alert.
     * 
     * @return Get the AlertState of the last time this Alert was raised.
     */
    public AlertState getLatestAlertState() {
        int n = getNumberOfInstances();
        return switch (n) {
            case 0 -> AlertState.NOMINAL;
            default -> instances.get(n-1).getAlertState();                
        };
    }
    
    /**
     * Get the latest CCS timestamp of this Alert.
     * @return CCS timestamp of the last time this Alert was raised.
     */
    public CCSTimeStamp getLatestAlertCCSTimeStamp() {
        int n = getNumberOfInstances();
        return switch (n) {
            case 0 -> null;
            default -> instances.get(n-1).getCCSTimeStamp();
        };
    }
    
    /**
     * Gets the latest cause of this Alert.
     * @return Cause that was associated to the latest instance of this alert.
     */
    public String getLatestAlertCause() {
        int n = getNumberOfInstances();
        return switch (n) {
            case 0 -> "";
            default -> instances.get(n-1).getCause();
        };
    }
    
    /**
     * Gets the latest raised instance of this alert.
     * @return {@code RaisedAlertInstance} depicting the last time this alert was raised.
     */
    public RaisedAlertInstance getLatestAlertInstance() {
        int n = getNumberOfInstances();
        return switch(n) {
            case 0 -> null;
            default -> instances.get(n-1);
        };
    }
    
    /**
     * Get the highest {@code AlertState} raised for this Alert.
     * @return The highest AlertState for this Alert.
     */
    public AlertState getHighestAlertState() {
        return highestSeverity;
    }
    
    /**
     * Get the Alert that was raised with the highest severity.
     * @return The highest Alert for this Alert.
     */
    public Alert getHighestAlert() {
        return highestSeverityAlert;
    }

    /**
     * Get the full history of RaisedAlertInstances for this Alert.
     * @return List of {@code RaisedAlertInstance}s for this history, in increasing time order.
     */
    public ArrayList<RaisedAlertInstance> getRaisedAlertInstancesList() {
        return new ArrayList<>(instances);
    }
    
    /**
     * Has this alert been acknowledged?
     * @return True if this alert is currently acknowledged.
     */
    public boolean isAcknowledged() {
        return ack != null;
    }
    
    /**
     * Returns an acknowledgement associated with this alert ID.
     * @return Acknowledgement, or {@code null} if this alert is not currently acknowledged.
     */
    public Acknowledgement getAcknowledgement() {
        return ack;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
            sb.append("Raised Alarms for id ").append(getLatestAlert().getAlertId()).append("\n");
            sb.append("Overall Severity: ").append(getHighestAlertState());
        return sb.toString();
    }

    public void setHighestSeverit(AlertState highestSeverity) {
        this.highestSeverity = highestSeverity;
    }
    
    static public class Acknowledgement implements Serializable {
        
        public final String message;
        public final String user;
        public final long time;
        public final AlertState highestSeverity;
        public final AlertState latestSeverity;
        
        public Acknowledgement(String message, String user, RaisedAlertHistory history) {
            this.message = message;
            this.user = user;
            time = System.currentTimeMillis();
            highestSeverity = history.getHighestAlertState();
            latestSeverity = history.getLatestAlertState();
        }

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("Acknowledged by ").append(user).append(". ").append(message);
            return sb.toString();
        }
    }
        
}
